package ru.yandex.ljinx;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.collection.Pattern;
import ru.yandex.function.GenericBiConsumer;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.server.async.DelegatedHttpAsyncRequestHandler;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.parser.config.ConfigException;

public class Ljinx extends HttpProxy<ImmutableLjinxConfig> {
    private final List<ProxyPassHandler> proxyPassHandlers = new ArrayList<>();
    private final Map<String, CacheStorage> cacheStorages;

    public Ljinx(final ImmutableLjinxConfig config)
        throws ConfigException, IOException, CacheStorageException
    {
        super(config);
        cacheStorages = config.cacheStoragesConfig().prepareStorages(this);
        config.proxyPassesConfig().proxyPasses()
            .traverse(new ProxyPassVisitor());
        register(
            new Pattern<>("/reloadsearchmap", false),
            new DelegatedHttpAsyncRequestHandler<HttpRequest>(
                new SearchMapReloader(),
                this,
                executor),
            RequestHandlerMapper.GET);
        for (CacheStorage cacheStorage: cacheStorages.values()) {
            registerStater(cacheStorage);
        }
    }

    public CacheStorage cacheStorage(final String storageName) {
        return cacheStorages.get(storageName);
    }

    @Override
    public void start() throws IOException {
        for (CacheStorage storage: cacheStorages.values()) {
            storage.start();
        }
        super.start();
    }

    @Override
    public void close() throws IOException {
        super.close();
        for (CacheStorage storage: cacheStorages.values()) {
            storage.close();
        }
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        for (Map.Entry<String, CacheStorage> entry: cacheStorages.entrySet()) {
            status.put(
                "cache-storage-" + entry.getKey(),
                entry.getValue().status(verbose));
        }
        return status;
    }

    private class ProxyPassVisitor
        implements GenericBiConsumer<
            Pattern<RequestInfo>,
            ImmutableProxyPassConfig,
            IOException>
    {
        @Override
        public void accept(
            final Pattern<RequestInfo> pattern,
            final ImmutableProxyPassConfig config)
            throws IOException
        {
            ProxyPassHandler handler =
                new ProxyPassHandler(config, Ljinx.this, pattern.toString());
            proxyPassHandlers.add(handler);
            register(pattern, handler);
            logger().info("Register: " + pattern);
        }
    }

    private class SearchMapReloader
        implements HttpAsyncRequestHandler<HttpRequest>
    {
        @Override
        public void handle(
            final HttpRequest request,
            final HttpAsyncExchange exchange,
            final HttpContext context)
            throws HttpException
        {
            Logger logger = (Logger) context.getAttribute(LOGGER);
            try {
                for (ProxyPassHandler handler: proxyPassHandlers) {
                    logger.info("Reloading searchmap for " + handler);
                    handler.reloadSearchMap();
                }
            } catch (IOException | ParseException e) {
                logger.log(Level.WARNING, "Searchmap reload failed", e);
                throw new ServiceUnavailableException(e);
            }
            logger.info("Searchmaps reload completed");
            exchange.getResponse().setEntity(
                new NStringEntity(
                    "Searchmaps reloaded",
                    ContentType.TEXT_PLAIN));
            exchange.submitResponse();
        }

        @Override
        public BasicAsyncRequestConsumer processRequest(
            final HttpRequest request,
            final HttpContext context)
        {
            return new BasicAsyncRequestConsumer();
        }

        @Override
        public String toString() {
            return "Reloads searchmap from file";
        }
    }
}
