package ru.yandex.ljinx;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.regex.Pattern;

import org.apache.http.FormattedHeader;
import org.apache.http.Header;
import org.apache.http.HeaderIterator;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.protocol.HTTP;
import org.apache.http.util.CharArrayBuffer;

import ru.yandex.charset.StreamEncoder;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.NByteArrayEntityGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.function.RequestFunctionValue;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.io.HexOutputStream;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.util.string.HexStrings;
import ru.yandex.util.string.PrettyPrint;
import ru.yandex.util.string.UnhexStrings;
import ru.yandex.util.timesource.TimeSource;

public class LuceneCacheStorage extends MemoryCacheStorage {
    public static final int IOPRIO = 3000;

    private static final String CACHE_KEY = "cache_key";
    private static final String DOCS = "docs";
    private static final String PREFIX = "prefix";
    private static final String HTTP_BODY = "http_body";
    private static final String HTTP_HEADERS = "http_headers";
    private static final String HTTP_STATUS = "http_status";
    private static final String HTTP_EXPIRE_TIMESTAMP = "http_expire_timestamp";

    private static final Pattern LUCENE_QUOTER =
        Pattern.compile("[:\\[\\]\\{\\}\\(\\)\\\\ ]");

    protected final long defaultTTL;
    protected final long minimalStoreTTL;

    private final ConcurrentHashMap<
        RequestFunctionValue,
        BasicAsyncResponseProducerGenerator> tempStoreMap =
            new ConcurrentHashMap<>();

    private final AsyncClient searchClient;
    private final AsyncClient indexClient;
    private final HttpHost searchHost;
    private final HttpHost indexHost;
    private final int luceneShards;
    private final boolean loadHitsToMemory;
    private final String primaryKeyField;

    public LuceneCacheStorage(
        final String storageName,
        final Ljinx ljinx,
        final LuceneCacheStorageConfig config)
    {
        super(storageName, ljinx.config(), config);
        searchClient =
            ljinx.client("Search-" + storageName, config.searchConfig());
        indexClient =
            ljinx.client("Index-" + storageName, config.indexConfig());
        searchHost = config.searchConfig().host();
        indexHost = config.indexConfig().host();
        luceneShards = config.luceneShards();
        defaultTTL = config.defaultTTL();
        minimalStoreTTL = config.minimalStoreTTL();
        loadHitsToMemory = config.loadHitsToMemory();
        primaryKeyField = config.primaryKeyField();
    }

    private String quoteQuery(final String query) {
        return LUCENE_QUOTER.matcher(query).replaceAll("\\\\$0");
    }

    protected boolean loadHitsToMemory() {
        return loadHitsToMemory;
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public void get(
        final ProxyPassSession session,
        final FutureCallback<CacheResponse> callback)
    {
        final RequestFunctionValue key = session.cacheKey();
        CacheResponse memoryCached = super.get(session, key);
        if (memoryCached != null) {
            callback.completed(memoryCached);
            return;
        }
        BasicAsyncResponseProducerGenerator response = tempStoreMap.get(key);
        if (response != null) {
            callback.completed(
                new CacheResponse(
                    response,
                    CacheResponse.CacheType.MEMORY,
                    TimeSource.INSTANCE.currentTimeMillis() + ttl(session)));
            return;
        }
        final StringBuilder request = new StringBuilder();
        request.append("/search-ljinx?IO_PRIO=");
        request.append(IOPRIO);
        request.append("&check-copyness=false&prefix=");
        request.append(prefix(key));
        request.append("&text=");
        request.append(primaryKeyField);
        request.append(':');
        request.append(quoteQuery(session.cacheUrl()));
        request.append("&get=http_status,http_body,http_method,http_headers,"
            + "http_expire_timestamp,cache_key&json-type=dollar");
        final AsyncClient client =
            searchClient.adjust(session.session().context());
        Supplier<? extends HttpClientContext> contextGenerator =
            session.session().listener().createContextGeneratorFor(client);

        client.execute(
            searchHost,
            new BasicAsyncRequestProducerGenerator(new String(request)),
            SearchResultConsumerFactory.OK,
            contextGenerator,
            new SearchResultCallback(session, callback));
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public void put(
        final ProxyPassSession session,
        final BasicAsyncResponseProducerGenerator value,
        final FutureCallback<Void> callback)
    {
        put(session, value, value, callback);
    }

    protected long ttl(final ProxyPassSession session) {
        Long ttl = session.ttl();
        if (ttl == null) {
            ttl = defaultTTL;
        }
        return ttl;
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    protected void put(
        final ProxyPassSession session,
        final BasicAsyncResponseProducerGenerator memoryValue,
        final BasicAsyncResponseProducerGenerator luceneValue,
        final FutureCallback<Void> callback)
    {
        final boolean luceneStore;
        final long ttl = ttl(session);
        if (ttl <= minimalStoreTTL || luceneValue == null) {
            session.logger().info(
                "TTL too low for lucene store: " + ttl
                + " data will be stored only in memory");
            luceneStore = false;
        } else {
            luceneStore = true;
        }
        if (luceneStore) {
            tempStoreMap.put(session.cacheKey(), memoryValue);
        }
        long expireTimeStamp = TimeSource.INSTANCE.currentTimeMillis() + ttl;
        put(session.cacheKey(), memoryValue, expireTimeStamp);

        if (!luceneStore) {
            callback.completed(null);
            return;
        }

        final AsyncClient client =
            indexClient.adjust(session.session().context());
        Supplier<? extends HttpClientContext> contextGenerator =
            session.session().listener().createContextGeneratorFor(client);

        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        try (JsonWriter writer =
                new DollarJsonWriter(
                    new StreamEncoder(out, client.requestCharset())))
        {
            HttpResponse httpMessage = luceneValue.get().generateResponse();
            writer.startObject();
            writer.key(PREFIX);
            writer.value(prefix(session.cacheKey()));
            writer.key(DOCS);
            writer.startArray();
            writer.startObject();
            writer.key(primaryKeyField);
            writer.value(session.cacheUrl());
            writer.key(CACHE_KEY);
            writer.value(session.cacheKey().stringValue());
            writer.key(HTTP_STATUS);
            writer.value(httpMessage.getStatusLine().getStatusCode());
            writer.key(HTTP_BODY);
            HttpEntity entity = httpMessage.getEntity();
            writeEntity(entity, writer);
            writer.key(HTTP_HEADERS);
            writeHeaders(httpMessage.headerIterator(), entity, writer);
            writer.key(HTTP_EXPIRE_TIMESTAMP);
            writer.value(TimeUnit.MILLISECONDS.toSeconds(expireTimeStamp));
            writer.endObject();
            writer.endArray();
            writer.endObject();
        } catch (IOException e) {
            tempStoreMap.remove(session.cacheKey(), memoryValue);
            callback.failed(e);
            return;
        }
        client.execute(
            indexHost,
            new BasicAsyncRequestProducerGenerator(
                "/modify?journal=false&" + session.cacheUrl(),
                new NByteArrayEntityGenerator(
                    out,
                    ContentType.APPLICATION_JSON.withCharset(
                        client.requestCharset()))),
            EmptyAsyncConsumerFactory.ANY_GOOD,
            contextGenerator,
            new CleanupCallback(session, memoryValue, callback));
    }

    private String prefix(final RequestFunctionValue key) {
        return Integer.toString(Math.abs(key.hashCode() % luceneShards));
    }

    protected static boolean writeHeader(
        final boolean first,
        final Writer writer,
        final Header header)
        throws IOException
    {
        if (header == null) {
            return first;
        } else {
            if (!first) {
                writer.write('\n');
            }
            if (header instanceof FormattedHeader) {
                CharArrayBuffer buffer =
                    ((FormattedHeader) header).getBuffer();
                writer.write(buffer.buffer(), 0, buffer.length());
            } else {
                writer.write(header.getName());
                writer.write(':');
                writer.write(header.getValue());
            }
            return false;
        }
    }

    protected static void writeHeaders(
        final HeaderIterator headers,
        final HttpEntity entity,
        final JsonWriter writer)
        throws IOException
    {
        writer.startString();
        boolean first = true;
        while (headers.hasNext()) {
            first = writeHeader(first, writer, headers.nextHeader());
        }
        first = writeHeader(first, writer, entity.getContentType());
        writeHeader(first, writer, entity.getContentEncoding());
        writer.endString();
    }

    protected static void writeEntity(
        final HttpEntity entity,
        final JsonWriter writer)
        throws IOException
    {
        if (entity == null) {
            writer.nullValue();
        } else {
            writer.startString();
            entity.writeTo(new HexOutputStream(writer, HexStrings.UPPER));
            writer.endString();
        }
    }

    @SuppressWarnings("StringSplitter")
    protected static BasicAsyncResponseProducerGenerator prepareResponse(
        Map<String, String> headers)
    {
        final String[] headersArray =
            headers.get(HTTP_HEADERS).split("\n");
        List<Header> headersList = new ArrayList<>(headersArray.length);
        Header contentType = null;
        Header contentEncoding = null;
        for (String header : headersArray) {
            int sep = header.indexOf(':');
            String name = header.substring(0, sep);
            if (name.equalsIgnoreCase(HTTP.CONTENT_TYPE)) {
                contentType = HeaderUtils.createHeader(header);
            } else if (name.equalsIgnoreCase(HTTP.CONTENT_ENCODING)) {
                contentEncoding = HeaderUtils.createHeader(header);
            } else {
                headersList.add(HeaderUtils.createHeader(header));
            }
        }
        BasicAsyncResponseProducerGenerator response =
            new BasicAsyncResponseProducerGenerator(
                Integer.parseInt(headers.get(HTTP_STATUS)),
                new NByteArrayEntityGenerator(
                    UnhexStrings.unhex(headers.get(HTTP_BODY)),
                    contentType,
                    contentEncoding));
        for (Header header : headersList) {
            response.addHeader(header);
        }
        return response;
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put("search-client", searchClient.status(verbose));
        status.put("index-client", indexClient.status(verbose));
        return status;
    }

    private class SearchResultCallback
        extends AbstractProxySessionCallback<SearchResult>
    {
        private final ProxyPassSession ppSession;
        private final FutureCallback<CacheResponse> callback;

        SearchResultCallback(
            final ProxyPassSession ppSession,
            final FutureCallback<CacheResponse> callback)
        {
            super(ppSession.session());
            this.ppSession = ppSession;
            this.callback = callback;
        }

        private void parseDoc(final SearchDocument doc) {
            try {
                long expireTimeStamp =
                    TimeUnit.SECONDS.toMillis(
                        Long.parseLong(
                            doc.attrs().get(HTTP_EXPIRE_TIMESTAMP)));
                long now = TimeSource.INSTANCE.currentTimeMillis();
                if (expireTimeStamp < now) {
                    ppSession.logger().info(
                        "Cache entry for key <" + ppSession.cacheKey()
                        + "> is expired at " + expireTimeStamp);
                    callback.completed(null);
                    return;
                }
                BasicAsyncResponseProducerGenerator response =
                    prepareResponse(doc.attrs());
                if (loadHitsToMemory()) {
                    put(ppSession.cacheKey(), response, expireTimeStamp);
                }
                callback.completed(
                    new CacheResponse(
                        response,
                        CacheResponse.CacheType.DISK,
                        expireTimeStamp));
            } catch (RuntimeException e) {
                ppSession.logger().log(
                    Level.SEVERE,
                    "Error occured while trying to parse lucene doc",
                    e);
                callback.completed(null);
            }
        }

        @Override
        public void completed(final SearchResult result) {
            for (final SearchDocument doc : result.hitsArray()) {
                final String docKey = doc.attrs().get(CACHE_KEY);
                try {
                    if (docKey != null) {
                        if (ppSession.cacheKey().equalsString(docKey)) {
                            ppSession.logger().info("Got lucene doc with key: "
                                + PrettyPrint.shortenString(docKey));
                            parseDoc(doc);
                            return;
                        } else {
                            ppSession.logger().info(
                                "Skipping lucene doc with key: "
                                + PrettyPrint.shortenString(docKey));
                        }
                    } else {
                        ppSession.logger().warning("Null doc key");
                    }
                } catch (RuntimeException e) {
                    ppSession.logger().log(
                        Level.WARNING,
                        "Failed to deserialize doc key: " + docKey,
                        e);
                }
            }
            ppSession.logger().info("Lucene cache miss");
            callback.completed(null);
        }

        @Override
        public void failed(final Exception e) {
            ppSession.logger().log(
                Level.SEVERE,
                "Lucene request failed",
                e);
            callback.completed(null);
        }
    }

    private class CleanupCallback
        extends AbstractProxySessionCallback<Object>
    {
        private final ProxyPassSession ppSession;
        private final BasicAsyncResponseProducerGenerator value;
        private final FutureCallback<Void> callback;

        CleanupCallback(
            final ProxyPassSession ppSession,
            final BasicAsyncResponseProducerGenerator value,
            final FutureCallback<Void> callback)
        {
            super(ppSession.session());
            this.ppSession = ppSession;
            this.callback = callback;
            this.value = value;
        }

        @Override
        public void completed(final Object unused) {
            tempStoreMap.remove(ppSession.cacheKey(), value);
            callback.completed(null);
        }

        @Override
        public void failed(final Exception e) {
            tempStoreMap.remove(ppSession.cacheKey(), value);
            callback.failed(e);
        }
    }
}
