package ru.yandex.ljinx;

import java.util.concurrent.TimeUnit;

import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.NonEmptyValidator;

public class LuceneCacheStorageConfig extends MemoryCacheStorageConfig {
    public static final int DEFAULT_LUCENE_SHARDS = 100;
    public static final long DEFAULT_MINIMAL_TTL =
        TimeUnit.MINUTES.toMillis(1L);
    public static final LuceneCacheStorageConfig DEFAULTS =
        new LuceneCacheStorageConfig();

    private final ImmutableHttpHostConfig searchConfig;
    private final ImmutableHttpHostConfig indexConfig;
    private final int luceneShards;
    private final long minimalStoreTTL;
    private final boolean loadHitsToMemory;
    private final String primaryKeyField;

    protected LuceneCacheStorageConfig() {
        super();
        luceneShards = DEFAULT_LUCENE_SHARDS;
        searchConfig = null;
        indexConfig = null;
        minimalStoreTTL = DEFAULT_MINIMAL_TTL;
        loadHitsToMemory = false;
        primaryKeyField = "url";
    }

    public LuceneCacheStorageConfig(
        final IniConfig config,
        final LuceneCacheStorageConfig other)
        throws ConfigException
    {
        super(config, other);
        LuceneCacheStorageConfig defaults = other;
        if (defaults == null) {
            defaults = DEFAULTS;
        }
        searchConfig =
            new HttpHostConfigBuilder(
                config.section("search")).build();
        indexConfig =
            new HttpHostConfigBuilder(
                config.section("index")).build();
        luceneShards = config.getInt(
            "lucene-shards",
            defaults.luceneShards());
        minimalStoreTTL = config.get(
            "minimal-store-ttl",
            defaults.minimalStoreTTL(),
            DurationParser.POSITIVE_LONG);
        loadHitsToMemory = config.getBoolean(
            "load-hits-to-memory",
            defaults.loadHitsToMemory());
        primaryKeyField = config.get(
            "primary-key-field",
            defaults.primaryKeyField(),
            NonEmptyValidator.INSTANCE);
    }

    public ImmutableHttpHostConfig searchConfig() {
        return searchConfig;
    }

    public ImmutableHttpHostConfig indexConfig() {
        return indexConfig;
    }

    public int luceneShards() {
        return luceneShards;
    }

    public long minimalStoreTTL() {
        return minimalStoreTTL;
    }

    public boolean loadHitsToMemory() {
        return loadHitsToMemory;
    }

    public String primaryKeyField() {
        return primaryKeyField;
    }

    @Override
    public CacheStorageType type() {
        return CacheStorageType.LUCENE;
    }

    @Override
    public CacheStorage createStorage(final Ljinx ljinx, final String name)
        throws CacheStorageException
    {
        return new LuceneCacheStorage(name, ljinx, this);
    }
}
