package ru.yandex.ljinx;

import ru.yandex.client.tvm2.ImmutableTvm2ClientConfig;
import ru.yandex.client.tvm2.ImmutableTvm2ServiceConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.client.tvm2.Tvm2ServiceConfig;
import ru.yandex.client.tvm2.Tvm2ServiceConfigBuilder;
import ru.yandex.http.config.DnsConfigBuilder;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.NonEmptyValidator;

public class MdsCacheStorageConfig extends LuceneCacheStorageConfig {
    public static final MdsCacheStorageConfig DEFAULTS = new MdsCacheStorageConfig();

    private static final String TVM2 = "tvm2";

    private final ImmutableHttpHostConfig backendWriterConfig;
    private final ImmutableHttpHostConfig backendReaderConfig;
    private final ImmutableDnsConfig dnsConfig;
    private final long mdsDefaultTTL;
    private final String mdsNamespace;

    private ImmutableTvm2ClientConfig tvm2ClientConfig;
    private ImmutableTvm2ServiceConfig tvm2ServiceConfig;

    protected MdsCacheStorageConfig() {
        super();
        backendWriterConfig = null;
        backendReaderConfig = null;
        dnsConfig = null;
        tvm2ServiceConfig = null;
        tvm2ClientConfig = null;
        mdsDefaultTTL = defaultTTL();
        mdsNamespace = "ps-cache";
    }

    public MdsCacheStorageConfig(
        final IniConfig config,
        final MdsCacheStorageConfig other)
        throws ConfigException
    {
        super(config, other);
        MdsCacheStorageConfig defaults = other;
        if (defaults == null) {
            defaults = DEFAULTS;
        }
        backendWriterConfig =
            new HttpHostConfigBuilder(config.section("to-write")).build();
        backendReaderConfig =
            new HttpHostConfigBuilder(config.section("to-read")).build();
        dnsConfig = new DnsConfigBuilder(config.section("dns")).build();
        tvm2ServiceConfig =
            new Tvm2ServiceConfigBuilder(config.section(TVM2)).build();
        tvm2ClientConfig =
            new Tvm2ClientConfigBuilder(config.section(TVM2)).build();
        mdsDefaultTTL = config.get(
            "store-ttl",
            defaults.mdsDefaultTTL(),
            DurationParser.POSITIVE_LONG);
        mdsNamespace = config.get(
            "namespace",
            defaults.mdsNamespace(),
            NonEmptyValidator.INSTANCE);
    }

    public ImmutableHttpHostConfig backendWriterConfig() {
        return backendWriterConfig;
    }

    public ImmutableHttpHostConfig backendReaderConfig() {
        return backendReaderConfig;
    }

    public ImmutableDnsConfig dnsConfig() {
        return dnsConfig;
    }

    public ImmutableTvm2ServiceConfig tvm2ServiceConfig() {
        return tvm2ServiceConfig;
    }

    public void tvm2ServiceConfig(final Tvm2ServiceConfig tvm2ServiceConfig)
        throws ConfigException
    {
        this.tvm2ServiceConfig =
            new ImmutableTvm2ServiceConfig(tvm2ServiceConfig);
    }

    public ImmutableTvm2ClientConfig tvm2ClientConfig() {
        return tvm2ClientConfig;
    }

    public void tvm2ClientConfig(final Tvm2ClientConfig tvm2ClientConfig)
        throws ConfigException
    {
        this.tvm2ClientConfig = new ImmutableTvm2ClientConfig(tvm2ClientConfig);
    }

    public String mdsTvmClientId() {
        return tvm2ClientConfig.destinationClientId();
    }

    public long mdsDefaultTTL() {
        return mdsDefaultTTL;
    }

    public String mdsNamespace() {
        return mdsNamespace;
    }

    @Override
    public CacheStorageType type() {
        return CacheStorageType.MDS;
    }

    @Override
    public CacheStorage createStorage(final Ljinx ljinx, final String name)
        throws CacheStorageException
    {
        return new MdsCacheStorage(name, ljinx, this);
    }
}
