package ru.yandex.stater;

import java.util.ArrayList;
import java.util.List;

import ru.yandex.parser.config.ConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.NonEmptyValidator;

public abstract class AbstractStaterConfigBuilder
    <T extends AbstractStaterConfigBuilder<T>>
    implements ConfigBuilder<T>, StaterConfig
{
    private String prefix;
    private List<MetricBuilder> metrics;
    private long metricsTimeFrame;
    private String category;
    private String title;

    protected AbstractStaterConfigBuilder(final StaterConfig config) {
        prefix(config.prefix());
        metrics(config.metrics());
        metricsTimeFrame(config.metricsTimeFrame());
        category(config.category());
        title(config.title());
    }

    @SuppressWarnings("StringSplitter")
    protected AbstractStaterConfigBuilder(
        final IniConfig config,
        final StaterConfig defaults)
        throws ConfigException
    {
        prefix = config.getString("prefix", defaults.prefix());
        final String metricsString = config.getString("metrics", null);
        if (metricsString == null) {
            this.metrics = defaults.metrics();
        } else {
            final String[] metricArray = metricsString.split("[,; ]");
            this.metrics = new ArrayList<>(metricArray.length);
            for (String metricString : metricArray) {
                final String metricName = metricString.trim();
                if (metricName.length() != 0) {
                    metrics.add(MetricsFactory.create(metricName, config));
                } // else empty metrics, i.e. disable metrics for this section
            }
        }
        metricsTimeFrame = config.get(
            "metrics-time-frame",
            defaults.metricsTimeFrame(),
            DurationParser.POSITIVE_LONG);
        category = config.get(
            "category",
            defaults.category(),
            NonEmptyValidator.TRIMMED);
        title = config.get(
            "title",
            defaults.title(),
            NonEmptyValidator.TRIMMED);
    }

    @Override
    public String prefix() {
        return prefix;
    }

    public T prefix(final String prefix) {
        this.prefix = prefix;
        return self();
    }

    @Override
    public List<MetricBuilder> metrics() {
        return metrics;
    }

    public T metrics(final List<MetricBuilder> metrics) {
        this.metrics = metrics;
        return self();
    }

    @Override
    public long metricsTimeFrame() {
        return metricsTimeFrame;
    }

    public T metricsTimeFrame(final long metricsTimeFrame) {
        this.metricsTimeFrame = metricsTimeFrame;
        return self();
    }

    @Override
    public String category() {
        return category;
    }

    public T category(final String category) {
        this.category = category;
        return self();
    }

    @Override
    public String title() {
        return title;
    }

    public T title(final String title) {
        this.title = title;
        return self();
    }
}

