package ru.yandex.stater;

import java.util.EnumMap;
import java.util.Locale;
import java.util.function.UnaryOperator;

import ru.yandex.util.string.StringUtils;

public class EnumStater<T extends Enum<T>>
    extends EnumMap<T, int[]>
    implements PassiveStater<T>
{
    private static final long serialVersionUID = 0L;

    private final transient UnaryOperator<String> enumToSignal;
    private final String signalPrefix;
    private final String category;
    private final String name;
    private final String title;
    private final Integer fractionSize;

    public EnumStater(
        final Class<T> clazz,
        final UnaryOperator<String> enumToSignal,
        final T[] possibleValues,
        final String signalPrefix,
        final String category,
        final String name,
        final String title,
        final Integer fractionSize)
    {
        super(clazz);
        this.enumToSignal = enumToSignal;
        for (T t: possibleValues) {
            put(t, new int[1]);
        }
        this.signalPrefix = signalPrefix;
        this.category = category;
        this.name = name;
        this.title = title;
        this.fractionSize = fractionSize;
    }

    @Override
    public void accept(final T t) {
        ++get(t)[0];
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        int total = 0;
        for (Entry<T, int[]> entry: entrySet()) {
            int count = entry.getValue()[0];
            total += count;
            statsConsumer.stat(
                signalPrefix
                + enumToSignal.apply(
                    entry.getKey()
                        .name()
                        .toLowerCase(Locale.ROOT)
                        .replace('_', '-')),
                count);
        }
        statsConsumer.stat(signalPrefix + enumToSignal.apply("total"), total);
    }

    @Override
    public boolean equals(final Object o) {
        return (o instanceof EnumStater)
            && enumToSignal.equals(((EnumStater<?>) o).enumToSignal)
            && super.equals(o);
    }

    @Override
    public int hashCode() {
        return enumToSignal.hashCode() ^ super.hashCode();
    }

    @Override
    public void addToGolovanPanel(
        final GolovanPanel panel,
        final String statsPrefix)
    {
        if (category == null || name == null) {
            return;
        }

        String prefix = StringUtils.removeSuffix(signalPrefix, '-');
        String chartsPrefix = statsPrefix + prefix;
        GolovanChartGroup group =
            new GolovanChartGroup(chartsPrefix, statsPrefix);
        ImmutableGolovanPanelConfig config = panel.config();

        GolovanChart distribution = new GolovanChart(
            "-distribution",
            ' ' + name + " distribution (%)",
            true,
            false,
            0d);
        String totalName = chartsPrefix + '-' + enumToSignal.apply("total");
        for (T value: keySet()) {
            String name =
                value.name().toLowerCase(Locale.ROOT).replace('_', '-');
            distribution.addSignal(
                new GolovanSignal(
                    "perc(" + chartsPrefix + '-' + enumToSignal.apply(name)
                    + ',' + totalName + ')',
                    config.tag(),
                    name,
                    null,
                    fractionSize,
                    true));
        }
        group.addChart(distribution);

        GolovanChart totals = new GolovanChart(
            "-total",
            ' ' + name + " total per " + config.splitBy() + " (rps)",
            false,
            true,
            0d);
        totals.addSplitSignal(
            config,
            totalName,
            fractionSize,
            false,
            false);
        group.addChart(totals);

        panel.addCharts(category, title, group);
    }
}

