package ru.yandex.stater;

import java.util.Collection;
import java.util.Map;
import java.util.function.IntFunction;

import ru.yandex.util.string.StringUtils;

public class FixedSetStater<T> implements Stater {
    private static final String AMMM = "_ammm";

    private final Iterable<T> data;
    private final IntFunction<? extends Map<T, long[]>> mapFactory;
    private final Object[] possibleValues;
    private final String signalPrefix;
    private final String category;
    private final String name;
    private final String title;
    private final String otherSignal;
    private final String totalSignal;
    private final Integer fractionSize;

    public FixedSetStater(
        final Iterable<T> data,
        final IntFunction<? extends Map<T, long[]>> mapFactory,
        final Collection<T> possibleValues,
        final String signalPrefix,
        final String category,
        final String name,
        final String title,
        final Integer fractionSize)
    {
        this.data = data;
        this.mapFactory = mapFactory;
        this.possibleValues = possibleValues.toArray();
        this.signalPrefix = signalPrefix;
        this.category = category;
        this.name = name;
        this.title = title;
        this.fractionSize = fractionSize;
        otherSignal = StringUtils.concat(signalPrefix, "other", AMMM);
        totalSignal = StringUtils.concat(signalPrefix, "total", AMMM);
    }

    @SuppressWarnings("unchecked")
    private Map<T, long[]> initMap() {
        Map<T, long[]> map = mapFactory.apply(possibleValues.length);
        for (Object possibleValue: possibleValues) {
            map.put((T) possibleValue, new long[1]);
        }
        return map;
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        Map<T, long[]> map = initMap();
        long total = 0L;
        long other = 0L;
        for (T value: data) {
            ++total;
            long[] counter = map.get(value);
            if (counter == null) {
                ++other;
            } else {
                ++counter[0];
            }
        }
        for (Map.Entry<T, long[]> entry: map.entrySet()) {
            statsConsumer.stat(
                StringUtils.concat(
                    signalPrefix,
                    entry.getKey().toString(),
                    AMMM),
                entry.getValue()[0]);
        }
        statsConsumer.stat(otherSignal, other);
        statsConsumer.stat(totalSignal, total);
    }

    @Override
    public void addToGolovanPanel(
        final GolovanPanel panel,
        final String statsPrefix)
    {
        String prefix = StringUtils.removeSuffix(signalPrefix, '-');
        String chartsPrefix = statsPrefix + prefix;
        GolovanChartGroup group =
            new GolovanChartGroup(chartsPrefix, statsPrefix);
        ImmutableGolovanPanelConfig config = panel.config();

        GolovanChart distribution = new GolovanChart(
            "-distribution",
            ' ' + name + " distribution (rps)",
            true,
            true,
            0d);
        for (Object possibleValue: possibleValues) {
            String name = possibleValue.toString();
            distribution.addSignal(
                new GolovanSignal(
                    chartsPrefix + '-' + name + AMMM,
                    config.tag(),
                    name,
                    null,
                    fractionSize,
                    false));
        }
        distribution.addSignal(
            new GolovanSignal(
                statsPrefix + otherSignal,
                config.tag(),
                "other",
                "#ff0000",
                fractionSize,
                false));
        group.addChart(distribution);

        GolovanChart totals = new GolovanChart(
            "-total",
            ' ' + name + " total per " + config.splitBy() + " (rps)",
            false,
            true,
            0d);
        totals.addSplitSignal(
            config,
            statsPrefix + totalSignal,
            fractionSize,
            false,
            false);
        group.addChart(totals);

        panel.addCharts(category, title, group);
    }
}

