package ru.yandex.stater;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import ru.yandex.json.writer.JsonWriterBase;

public class GolovanChart {
    private static final Map<String, String> DC_COLORS = Map.of(
        "myt", "#37bff2",
        "iva", "#169833",
        "vla", "#f6ab31",
        "sas", "#c95edd");

    private final List<GolovanSignal> signals = new ArrayList<>();
    private final String idSuffix;
    private final String titleSuffix;
    private final boolean stacked;
    private final boolean normalize;
    private final Double minValue;

    public GolovanChart(
        final String idSuffix,
        final String titleSuffix,
        final boolean stacked,
        final boolean normalize,
        final Double minValue)
    {
        this.idSuffix = idSuffix;
        this.titleSuffix = titleSuffix;
        this.stacked = stacked;
        this.normalize = normalize;
        this.minValue = minValue;
    }

    public boolean isEmpty() {
        return signals.isEmpty();
    }

    public String idSuffix() {
        return idSuffix;
    }

    public void addSignal(final GolovanSignal signal) {
        signals.add(signal);
    }

    public void addSplitSignal(
        final ImmutableGolovanPanelConfig config,
        final String signal,
        final Integer fractionSize,
        final boolean drawAll,
        final boolean drawNines)
    {
        for (Map.Entry<String, String> location: config.splitValues()) {
            addSignal(
                new GolovanSignal(
                    signal,
                    config.splitTag() + ';'
                    + config.splitBy() + '=' + location.getKey(),
                    location.getValue(),
                    DC_COLORS.get(location.getValue()),
                    fractionSize,
                    false));
        }
        if (drawAll) {
            addSignal(
                new GolovanSignal(
                    signal,
                    config.tag(),
                    "all",
                    "#7f7f7f",
                    fractionSize,
                    false));
        }
        if (drawNines) {
            addSignal(
                new GolovanSignal(
                    "const(0.01)",
                    config.tag(),
                    "0.01%",
                    "#ffff00",
                    3,
                    false));
            addSignal(
                new GolovanSignal(
                    "const(0.1)",
                    config.tag(),
                    "0.1%",
                    "#ff0000",
                    3,
                    false));
        }
    }

    public void writeValue(
        final JsonWriterBase writer,
        final String idPrefix,
        final String titlePrefix,
        final int row,
        final int col)
        throws IOException
    {
        writer.startObject();
        writer.key("type");
        writer.value("graphic");
        writer.key("height");
        writer.value(1);
        writer.key("width");
        writer.value(1);
        writer.key("row");
        writer.value(row);
        writer.key("col");
        writer.value(col);
        if (idSuffix != null) {
            writer.key("id");
            writer.value(idPrefix + idSuffix);
        }
        if (titleSuffix != null) {
            writer.key("title");
            writer.value(titlePrefix + titleSuffix);
        }
        if (stacked) {
            writer.key("stacked");
            writer.value(true);
        }
        if (normalize) {
            writer.key("normalize");
            writer.value(true);
        }
        if (minValue != null) {
            writer.key("minValue");
            writer.value(minValue.doubleValue());
        }
        writer.key("yAxis");
        writer.startArray();
        writer.startObject();
        writer.endObject();
        writer.endArray();
        writer.key("consts");
        writer.startArray();
        writer.endArray();
        writer.key("signals");
        writer.startArray();
        for (GolovanSignal signal: signals) {
            signal.writeValue(writer);
        }
        writer.endArray();
        writer.endObject();
    }
}

