package ru.yandex.stater;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import ru.yandex.collection.IntPair;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

public class GolovanPanel implements JsonValue {
    public static final List<String> GOOD_COLORS = List.of(
        "#003f00",
        "#007f00",
        "#00bf00",
        "#00ff00",
        // "3fff00", remarkable not contrasting with previous
        "#7fff00",
        "#bfff00");

    public static final List<String> WARNING_COLORS = List.of(
        "#ffff00",
        "#ffbf00",
        "#ff7f00");

    public static final List<String> BAD_COLORS = List.of(
        "#ff3f00",
        "#ff0000",
        "#bf0000",
        "#7f0000",
        "#3f0000",
        "#000000");

    public static final List<String> ALL_COLORS;

    static {
        List<String> colors = new ArrayList<>(
            GOOD_COLORS.size() + WARNING_COLORS.size() + BAD_COLORS.size());
        colors.addAll(GOOD_COLORS);
        colors.addAll(WARNING_COLORS);
        colors.addAll(BAD_COLORS);
        ALL_COLORS = Collections.unmodifiableList(colors);
    }

    // Map from category to chart groups,
    // When adding charts, find group with the same title
    // Create new chart group if there is no group with the same title
    private final Map<String, List<TitledGolovanChartsGroup>> categories =
        new LinkedHashMap<>();
    // Map from chart group id to title and chart order
    // Empty chart name means that this chart should be omitted
    private final Map<String, IntPair<String>> rename = new HashMap<>();
    private final ImmutableGolovanPanelConfig config;

    public GolovanPanel(final ImmutableGolovanPanelConfig config) {
        this.config = config;
        for (String category: config.categoriesOrder()) {
            categories.put(category, new ArrayList<>());
        }
        for (Map.Entry<String, String> entry: config.rename().entrySet()) {
            rename.put(
                entry.getKey(),
                new IntPair<>(rename.size(), entry.getValue()));
        }
    }

    public ImmutableGolovanPanelConfig config() {
        return config;
    }

    public void addCharts(
        final String category,
        final String title,
        final GolovanChartGroup charts)
    {
        if (charts.isEmpty()) {
            return;
        }

        String realCategory = config.renameCategories().get(category);
        if (realCategory == null) {
            realCategory = category;
        }
        List<TitledGolovanChartsGroup> groups = categories.get(realCategory);
        if (groups == null) {
            return;
        }
        for (TitledGolovanChartsGroup group: groups) {
            if (Objects.equals(title, group.title())) {
                group.addCharts(charts);
                return;
            }
        }
        TitledGolovanChartsGroup group =
            new TitledGolovanChartsGroup(title, config.maxCols(), rename);
        group.addCharts(charts);
        groups.add(group);
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writer.key("type");
        writer.value("panel");
        writer.key("abc");
        writer.value(config.abc());
        writer.key("title");
        writer.value(config.title());
        writer.key("editors");
        writer.value(config.editors());
        writer.key("charts");
        writer.startArray();
        int row = 1;
        for (List<TitledGolovanChartsGroup> groups: categories.values()) {
            for (TitledGolovanChartsGroup group: groups) {
                row += group.writeValue(writer, row);
            }
        }
        writer.endArray();
        writer.endObject();
    }

    public static List<String> colors(
        final List<String> palette,
        final int count)
    {
        List<String> colors = new ArrayList<>(count);
        int size = palette.size();
        if (count >= size) {
            colors.addAll(palette);
            while (size++ < count) {
                colors.add(null);
            }
        } else {
            // step from the boundary colors if possible
            int stepDivisor = count + 1;
            if (size / stepDivisor <= 1) {
                stepDivisor = count;
            }
            int off = (size - (count - 1) * size / stepDivisor) / 2;
            for (int i = 0; i < count; ++i) {
                colors.add(palette.get(i * size / stepDivisor + off));
            }
        }
        return colors;
    }
}

