package ru.yandex.stater;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.util.string.StringUtils;

public class HttpCodesMetric implements Metric, MetricBuilder {
    private static final int SEC_MSEC = 1000;
    private static final String TOTAL = "total";

    private final String prefix;
    private final CodeInterval[] codes;
    private final Set<CodeInterval> goodCodes;
    private final ImmutableHttpCodesAlertsConfig alertsConfig;
    private final ImmutableHttpCodesAlertsConfig disasterAlertsConfig;

    @SuppressWarnings("StringSplitter")
    public HttpCodesMetric(final IniConfig config) throws ConfigException {
        prefix = "";
        codes =
            parseCodeIntervals(
                config.getString(
                    "httpcode-ranges",
                    "total, 200-299, 400-499, 500-599"));
        String goodCodesRanges = config.getString("good-http-codes", null);
        if (goodCodesRanges == null) {
            List<CodeInterval> goodCodes = new ArrayList<>();
            for (CodeInterval code: codes) {
                if (code.start < 400 && code.start != 0) {
                    goodCodes.add(code);
                }
            }
            this.goodCodes = new LinkedHashSet<>(goodCodes);
        } else {
            goodCodes =
                new LinkedHashSet<>(
                    Arrays.asList(parseCodeIntervals(goodCodesRanges)));

            Set<CodeInterval> unusedGoodCodes = new LinkedHashSet<>(goodCodes);
            unusedGoodCodes.removeAll(
                new LinkedHashSet<>(Arrays.asList(codes)));
            if (!unusedGoodCodes.isEmpty()) {
                throw new ConfigException(
                    "The following good-http-codes don't have corresponding "
                    + "httpcode-ranges: " + unusedGoodCodes);
            }
        }

        IniConfig alert = config.sectionOrNull("alert");
        if (alert == null) {
            alertsConfig = null;
            disasterAlertsConfig = null;
        } else {
            alertsConfig = new HttpCodesAlertsConfigBuilder(alert).build();
            IniConfig disasterAlert = config.sectionOrNull("disaster-alert");
            if (disasterAlert == null) {
                disasterAlertsConfig = null;
            } else {
                disasterAlertsConfig =
                    new HttpCodesAlertsConfigBuilder(disasterAlert).build();
                GolovanAlertThreshold.Mode alertMode =
                    alertsConfig.errorsThreshold().mode();
                GolovanAlertThreshold.Mode disasterAlertMode =
                    disasterAlertsConfig.errorsThreshold().mode();
                if (alertMode != disasterAlertMode) {
                    throw new ConfigException(
                        "Different alert modes not supported. Alert mode: "
                        + alertMode
                        + ", disaster alert mode: " + disasterAlertMode);
                }
                if (alertsConfig.minValue() == null
                    && disasterAlertsConfig.minValue() != null)
                {
                    throw new ConfigException(
                        "Alert min-value must be set in order to use disaster "
                        + "alert min-value");
                }
            }
        }
    }

    private static CodeInterval[] parseCodeIntervals(
        final String rangesString)
    {
        String[] ranges = rangesString.split("[,;]");
        CodeInterval[] codes = new CodeInterval[ranges.length];
        for (int i = 0; i < ranges.length; ++i) {
            String range = ranges[i].trim();
            int len = range.length();
            int idx = range.indexOf('(');
            String displayName;
            if (idx == -1 || range.charAt(len - 1) != ')') {
                displayName = null;
            } else {
                displayName = range.substring(idx + 1, len - 1).trim();
                range = range.substring(0, idx).trim();
            }
            String[] startEnd = range.split("-");
            int start;
            int end;
            if (startEnd[0].equalsIgnoreCase(TOTAL)
                || startEnd[0].equalsIgnoreCase("all"))
            {
                start = 0;
                end = Integer.MAX_VALUE;
            } else {
                start = Integer.parseInt(startEnd[0]);
                if (startEnd.length == 1) {
                    end = start;
                } else {
                    end = Integer.parseInt(startEnd[1]);
                }
            }
            String name = makeName(start, end);
            if (displayName == null) {
                displayName = name;
            }
            codes[i] = new CodeInterval(start, end, name, displayName);
        }
        return codes;
    }

    public HttpCodesMetric(final HttpCodesMetric other, final String prefix) {
        this.prefix = prefix;
        codes = other.codes.clone();
        for (int i = 0; i < codes.length; i++) {
            codes[i] = other.codes[i].clone();
        }
        goodCodes = other.goodCodes;
        alertsConfig = other.alertsConfig;
        disasterAlertsConfig = other.disasterAlertsConfig;
    }

    @Override
    public HttpCodesMetric build(final String prefix) {
        return new HttpCodesMetric(this, prefix);
    }

    public static double round(final double value, final int places) {
        if (places < 0) {
            throw new IllegalArgumentException();
        }
        BigDecimal bd = new BigDecimal(value);
        bd = bd.setScale(places, RoundingMode.HALF_UP);
        return bd.doubleValue();
    }

    @Override
    public void accept(final RequestInfo info) {
        int code = info.httpCode();
        long requestLength = info.requestLength();
        long responseLength = info.responseLength();
        for (CodeInterval interval : codes) {
            interval.collect(code, requestLength, responseLength);
        }
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer,
        final int timeDiff)
        throws E
    {
        double secs = ((double) timeDiff) / SEC_MSEC;
        for (CodeInterval code : codes) {
            String name = prefix + '-' + code.name;
            long count = code.count;
            statsConsumer.stat(StringUtils.concat(name, "_ammm"), count);
            statsConsumer.stat(
                StringUtils.concat(name, "_rps_ammv"),
                round(count / secs, 2));
            statsConsumer.stat(
                StringUtils.concat(name, "-request-length_ammm"),
                code.requestLength);
            statsConsumer.stat(
                StringUtils.concat(name, "-response-length_ammm"),
                code.responseLength);
        }
    }

    private static void addIntervals(
        final GolovanChart chart,
        final String tag,
        final String prefix,
        final List<CodeInterval> intervals,
        final List<String> palette)
    {
        int size = intervals.size();
        Map<String, List<CodeInterval>> namedIntervals = new LinkedHashMap<>();
        for (int i = 0; i < size; ++i) {
            CodeInterval interval = intervals.get(i);
            List<CodeInterval> group =
                namedIntervals.remove(interval.displayName);
            if (group == null) {
                group = new ArrayList<>();
            }
            group.add(interval);
            namedIntervals.put(interval.displayName, group);
        }
        List<Map.Entry<String, List<CodeInterval>>> groupedIntervals =
            new ArrayList<>(namedIntervals.entrySet());
        size = groupedIntervals.size();
        List<String> colors = GolovanPanel.colors(palette, size);
        for (int i = size; i-- > 0;) {
            Map.Entry<String, List<CodeInterval>> group =
                groupedIntervals.get(i);
            String name = group.getKey();
            String shortName;
            if (name.startsWith("codes-")) {
                shortName = name.substring(6);
            } else {
                shortName = name;
            }
            List<CodeInterval> groupIntervals = group.getValue();
            int groupSize = groupIntervals.size();
            String signal;
            if (groupSize == 1) {
                signal = prefix + '-' + groupIntervals.get(0).name + "_ammm";
            } else {
                StringBuilder sb = new StringBuilder("sum(");
                for (int j = groupSize; j-- > 0;) {
                    sb.append(prefix);
                    sb.append('-');
                    sb.append(groupIntervals.get(j).name);
                    sb.append("_ammm");
                    if (j != 0) {
                        sb.append(',');
                    }
                }
                sb.append(')');
                signal = new String(sb);
            }
            chart.addSignal(
                new GolovanSignal(
                    signal,
                    tag,
                    shortName,
                    colors.get(size - i - 1),
                    1,
                    false));
        }
    }

    @Override
    public List<GolovanChart> createGolovanCharts(
        final ImmutableGolovanPanelConfig config,
        final String name)
    {
        GolovanChart rpsChart = new GolovanChart(
            "-rps",
            " RPS",
            true,
            true,
            0d);
        Intervals intervals = new Intervals(codes, goodCodes);
        addIntervals(
            rpsChart,
            config.tag(),
            name,
            intervals.bad,
            GolovanPanel.BAD_COLORS);
        addIntervals(
            rpsChart,
            config.tag(),
            name,
            intervals.warning,
            GolovanPanel.WARNING_COLORS);
        addIntervals(
            rpsChart,
            config.tag(),
            name,
            intervals.good,
            GolovanPanel.GOOD_COLORS);

        GolovanChart errorsChart = new GolovanChart(
            "-errors",
            " errors (%)",
            false,
            false,
            0d);
        errorsChart.addSplitSignal(
            config,
            "diff(const(100),perc(" + intervals.goodIntervalsSum(name)
            + ',' + name + "-total_ammm))",
            3,
            true,
            true);
        return Arrays.asList(rpsChart, errorsChart);
    }

    @Override
    public void addAlerts(
        final IniConfig alertsConfig,
        final ImmutableGolovanPanelConfig panelConfig,
        final String name)
        throws BadRequestException
    {
        if (this.alertsConfig != null) {
            ImmutableGolovanAlertsConfig alerts = panelConfig.alerts();
            String goodSum =
                new Intervals(codes, goodCodes).goodIntervalsSum(name);
            GolovanAlertThreshold errorsThreshold =
                this.alertsConfig.errorsThreshold();
            String signal;
            if (errorsThreshold.mode() == GolovanAlertThreshold.Mode.PERCENT) {
                signal =
                    "diff(const(100),perc(" + goodSum
                    + ',' + name + "-total_ammm))";
            } else {
                signal = "diff(" + name + "-total_ammm," + goodSum + ')';
            }
            AlertThresholds disasterAlertThresholds;
            if (disasterAlertsConfig == null) {
                disasterAlertThresholds = null;
            } else {
                disasterAlertThresholds =
                    new AlertThresholds(
                        disasterAlertsConfig.errorsThreshold().value(),
                        null,
                        disasterAlertsConfig.alertWindow());
            }
            alerts.createAlert(
                alertsConfig,
                alerts.module() + '-'
                + GolovanAlertsConfig.clearAlertName(name) + "-errors",
                signal,
                new AlertThresholds(
                    errorsThreshold.value(),
                    null,
                    this.alertsConfig.alertWindow()),
                disasterAlertThresholds);

            Integer minValue = this.alertsConfig.minValue();
            if (minValue != null) {
                if (disasterAlertsConfig == null) {
                    disasterAlertThresholds = null;
                } else {
                    Integer disasterMinValue = disasterAlertsConfig.minValue();
                    if (disasterMinValue == null) {
                        disasterAlertThresholds = null;
                    } else {
                        disasterAlertThresholds =
                            new AlertThresholds(
                                null,
                                disasterMinValue.doubleValue(),
                                null);
                    }
                }
                alerts.createAlert(
                    alertsConfig,
                    alerts.module() + '-'
                    + GolovanAlertsConfig.clearAlertName(name) + "-low",
                    goodSum,
                    new AlertThresholds(null, minValue.doubleValue(), null),
                    disasterAlertThresholds);
            }
        }
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("HttpCodesMetric[prefix=");
        sb.append(prefix);
        sb.append(" codes");
        char sep = '=';
        for (CodeInterval interval : codes) {
            sb.append(sep);
            sb.append(interval);
            sep = ',';
        }
        sb.append(']');
        return new String(sb);
    }

    private static String makeName(final int start, final int end) {
        if (start == 0 && end == Integer.MAX_VALUE) {
            return TOTAL;
        } else if (start == 0 && end == 99) {
            return "codes-0xx";
        } else if (start == 100 && end == 199) {
            return "codes-1xx";
        } else if (start == 200 && end == 299) {
            return "codes-2xx";
        } else if (start == 300 && end == 399) {
            return "codes-3xx";
        } else if (start == 400 && end == 499) {
            return "codes-4xx";
        } else if (start == 500 && end >= 599) {
            return "codes-5xx";
        } else if (start == end) {
            return Integer.toString(start);
        } else {
            return Integer.toString(start) + '-' + Integer.toString(end);
        }
    }

    private static class CodeInterval implements Cloneable {
        private final int start;
        private final int end;
        private final String name;
        private final String displayName;
        private long count = 0L;
        private long requestLength = 0L;
        private long responseLength = 0L;

        CodeInterval(
            final int start,
            final int end,
            final String name,
            final String displayName)
        {
            this.start = start;
            this.end = end;
            this.name = name;
            this.displayName = displayName;
        }

        public void collect(
            final int code,
            final long requestLength,
            final long responseLength)
        {
            if (code >= start && code <= end) {
                ++count;
                this.requestLength += requestLength;
                this.responseLength += responseLength;
            }
        }

        @Override
        public CodeInterval clone() {
            return new CodeInterval(start, end, name, displayName);
        }

        @Override
        public String toString() {
            return makeName(start, end);
        }

        @Override
        public int hashCode() {
            return start + (start ^ end);
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof CodeInterval) {
                CodeInterval other = (CodeInterval) o;
                return start == other.start && end == other.end;
            }
            return false;
        }
    }

    private static class Intervals {
        private final List<CodeInterval> good;
        private final List<CodeInterval> warning;
        private final List<CodeInterval> bad;

        Intervals(
            final CodeInterval[] codes,
            final Set<CodeInterval> goodCodes)
        {
            good = new ArrayList<>(codes.length);
            warning = new ArrayList<>(codes.length);
            bad = new ArrayList<>(codes.length);
            for (int i = 0; i < codes.length; ++i) {
                CodeInterval interval = codes[i];
                if (interval.name.equals(TOTAL)) {
                    continue;
                }
                if (goodCodes.contains(interval)) {
                    good.add(interval);
                } else if (interval.start < 499) {
                    warning.add(interval);
                } else {
                    bad.add(interval);
                }
            }
        }

        public String goodIntervalsSum(final String name) {
            int size = good.size();
            if (size == 1) {
                return name + '-' + good.get(0).name + "_ammm";
            } else {
                StringBuilder sb = new StringBuilder("sum(");
                for (int i = 0; i < size; ++i) {
                    if (i != 0) {
                        sb.append(',');
                    }
                    sb.append(name);
                    sb.append('-');
                    sb.append(good.get(i).name);
                    sb.append("_ammm");
                }
                sb.append(')');
                return new String(sb);
            }
        }
    }
}
