package ru.yandex.stater;

import java.util.LinkedHashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.olelole.config.ImmutableNotificationConfig;
import ru.yandex.olelole.config.NotificationConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.ImmutableConfig;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.util.string.StringUtils;

public class ImmutableGolovanAlertsConfig
    implements GolovanAlertsConfig, ImmutableConfig
{
    private final String namespace;
    private final String module;
    private final String wikiPrefix;
    private final String yasmPanel;
    private final String yasmTemplate;
    private final Integer stableTime;
    private final Integer critTime;
    private final Map<String, ImmutableNotificationConfig>
        notificationsConfigs;
    private final Map<String, ImmutableNotificationConfig>
        disasterNotificationsConfigs;

    public ImmutableGolovanAlertsConfig(final GolovanAlertsConfig config)
        throws ConfigException
    {
        namespace = NAMESPACE.validate(config.namespace());
        module = MODULE.validate(config.module());
        wikiPrefix = WIKI_PREFIX.validate(config.wikiPrefix());
        yasmPanel = YASM_PANEL.validate(config.yasmPanel());
        yasmTemplate = YASM_TEMPLATE.validate(config.yasmTemplate());
        stableTime = STABLE_TIME.validate(config.stableTime());
        critTime = CRIT_TIME.validate(config.critTime());

        Map<String, ? extends NotificationConfig> notificationsConfigs =
            config.notificationsConfigs();
        this.notificationsConfigs =
            new LinkedHashMap<>(notificationsConfigs.size() << 1);
        for (Map.Entry<String, ? extends NotificationConfig> entry
            : notificationsConfigs.entrySet())
        {
            String name = entry.getKey();
            this.notificationsConfigs.put(
                name,
                GolovanAlertsConfig.notificationConfigParser(name)
                    .validate(entry.getValue()));
        }

        Map<String, ? extends NotificationConfig>
            disasterNotificationsConfigs =
            config.disasterNotificationsConfigs();
        this.disasterNotificationsConfigs =
            new LinkedHashMap<>(disasterNotificationsConfigs.size() << 1);
        for (Map.Entry<String, ? extends NotificationConfig> entry
            : disasterNotificationsConfigs.entrySet())
        {
            String name = entry.getKey();
            if (notificationsConfigs.containsKey(name)) {
                throw new ConfigException(
                    "Disaster notification <" + name
                    + "> already presents in regular notifications");
            }
            this.disasterNotificationsConfigs.put(
                name,
                GolovanAlertsConfig.notificationConfigParser(name)
                    .validate(entry.getValue()));
        }

        if (stableTime == null) {
            if (critTime != null) {
                throw new ConfigException(
                    "stable-time is not set while crit-time is set");
            }
        } else if (critTime == null) {
            throw new ConfigException(
                "stable-time is set while crit-time is not set");
        } else if (stableTime.intValue() >= critTime.intValue()) {
            throw new ConfigException(
                "stable-time should be less than crit-time");
        }

        if (notificationsConfigs.isEmpty()) {
            throw new ConfigException("No notifications configured");
        }
    }

    @Override
    public String namespace() {
        return namespace;
    }

    @Override
    public String module() {
        return module;
    }

    @Override
    public String wikiPrefix() {
        return wikiPrefix;
    }

    @Override
    public String yasmPanel() {
        return yasmPanel;
    }

    @Override
    public String yasmTemplate() {
        return yasmTemplate;
    }

    @Override
    public Integer stableTime() {
        return stableTime;
    }

    @Override
    public Integer critTime() {
        return critTime;
    }

    @Override
    public Map<String, ImmutableNotificationConfig> notificationsConfigs() {
        return notificationsConfigs;
    }

    @Override
    public Map<String, ImmutableNotificationConfig>
        disasterNotificationsConfigs()
    {
        return disasterNotificationsConfigs;
    }

    public void createAlert(
        @Nonnull final IniConfig alertsConfig,
        @Nonnull final String alertName,
        @Nonnull final String signal,
        @Nonnull final AlertThresholds thresholds)
        throws BadRequestException
    {
        createAlert(alertsConfig, alertName, signal, thresholds, null);
    }

    public void createAlert(
        @Nonnull final IniConfig alertsConfig,
        @Nonnull final String alertName,
        @Nonnull final String signal,
        @Nonnull final AlertThresholds thresholds,
        @Nullable final AlertThresholds disasterThresholds)
        throws BadRequestException
    {
        IniConfig alertConfig = alertsConfig.section("alert." + alertName);
        if (disasterThresholds == null
            && !disasterNotificationsConfigs.isEmpty())
        {
            alertConfig.put(
                "notifications",
                StringUtils.join(disasterNotificationsConfigs.keySet(), ", "));
        } else {
            alertConfig.put(
                "notifications",
                StringUtils.join(notificationsConfigs.keySet(), ", "));
        }
        alertConfig.put("wiki", wikiPrefix + alertName);
        alertConfig.putAsString("stable-time", stableTime);
        alertConfig.putAsString("crit-time", critTime);
        alertConfig.put("signal", signal);
        thresholds.toIniConfig(alertConfig);

        if (disasterThresholds != null) {
            if (disasterNotificationsConfigs.isEmpty()) {
                throw new BadRequestException(
                    "Disaster threshold is set "
                    + "but not disaster notifications configured");
            }
            alertConfig =
                alertsConfig.section("alert." + alertName + "-disaster");
            alertConfig.put(
                "notifications",
                StringUtils.join(disasterNotificationsConfigs.keySet(), ", "));
            alertConfig.put("wiki", wikiPrefix + alertName);
            alertConfig.putAsString("stable-time", stableTime);
            alertConfig.putAsString("crit-time", critTime);
            alertConfig.put("signal", signal);
            disasterThresholds.toIniConfig(alertConfig);
        }
    }
}

