package ru.yandex.stater;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.util.string.StringUtils;

public class RequestTimeHistogramMetric
    implements Metric, MetricBuilder, PassiveStater<RequestInfo>
{
    private static final String INF = "inf";
    private static final String DEFAULT_TOPS =
        "0, 1, 10, 50, 100, 150, 200, 300, 400, 500, 600, 700, 800, 900, 1000,"
        + " 1500, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000,"
        + " 20000, 30000, 100000";

    private final NamingProvider namingProvider;
    private final String prefix;
    private final Backet[] backets;
    private final Backet[] processingBackets;
    private final boolean preciseHistogram;
    private final boolean processingTimeStats;

    public RequestTimeHistogramMetric(final IniConfig config)
        throws ConfigException
    {
        this(DefaultNamingProvider.INSTANCE, config);
    }

    public RequestTimeHistogramMetric(
        final NamingProvider namingProvider,
        final IniConfig config)
        throws ConfigException
    {
        this.namingProvider = namingProvider;
        prefix = "";
        preciseHistogram = config.getBoolean("precise-histogram", true);
        processingTimeStats = config.getBoolean("processing-time-stats", true);
        final String backetsString =
            config.getString(
                "histogram-ranges",
                DEFAULT_TOPS);
        if (backetsString.trim().length() == 0) {
            backets = new Backet[0];
            processingBackets = backets;
        } else {
            String[] backets = backetsString.split("[,;]");
            this.backets = new Backet[backets.length];
            for (int i = 0; i < backets.length; ++i) {
                String backet = backets[i];
                final long leftBound;
                if (backet.trim().equalsIgnoreCase(INF)
                    || backet.trim().equalsIgnoreCase("max"))
                {
                    leftBound = Long.MAX_VALUE;
                } else {
                    leftBound = Long.parseLong(backet.trim());
                }
                this.backets[i] = new Backet(leftBound);
            }
            Arrays.sort(backets);
            if (processingTimeStats) {
                processingBackets = copy(this.backets);
            } else {
                processingBackets = new Backet[0];
            }
        }
    }

    public RequestTimeHistogramMetric(
        final RequestTimeHistogramMetric other,
        final String prefix)
    {
        this.namingProvider = other.namingProvider;
        this.prefix = prefix;
        backets = copy(other.backets);
        processingBackets = copy(backets);
        preciseHistogram = other.preciseHistogram;
        processingTimeStats = other.processingTimeStats;
    }

    private static Backet[] copy(final Backet[] example) {
        Backet[] copy = new Backet[example.length];
        for (int i = 0; i < example.length; ++i) {
            copy[i] = new Backet(example[i].leftBound);
        }
        return copy;
    }

    @Override
    public RequestTimeHistogramMetric build(final String prefix) {
        return new RequestTimeHistogramMetric(this, prefix);
    }

    @Override
    public void accept(final RequestInfo info) {
        collectTimes(
            backets,
            info.requestTime());
        if (processingTimeStats) {
            collectTimes(
                processingBackets,
                info.processingTime());
        }
    }

    private void collectTimes(
        final Backet[] backets,
        final long time)
    {
        int len = backets.length;
        // left bound is exclusive for all histogram backets
        // except zero time backet
        // for precise histogram, left bound actually is inclusive right bound
        int i = 0;
        for (; i < len; ++i) {
            Backet backet = backets[i];
            if (time <= backet.leftBound) {
                ++backet.preciseCount;
                break;
            }
        }
        if (i == 0) {
            ++backets[0].count;
        } else {
            ++backets[i - 1].count;
        }
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer,
        final int timeDiff)
        throws E
    {
        stats(statsConsumer);
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        backets(statsConsumer, backets, "times");
        if (processingTimeStats) {
            backets(statsConsumer, processingBackets, "processing-times");
        }
    }

    private <E extends Exception> void backets(
        final StatsConsumer<? extends E> statsConsumer,
        final Backet[] backets,
        final String metricName)
        throws E
    {
        if (backets.length == 0) {
            return;
        }
        long[] hist = new long[backets.length << 1];
        int pos = 0;
        for (Backet backet: backets) {
            hist[pos++] = backet.leftBound;
            hist[pos++] = backet.count;
        }
        if (preciseHistogram) {
            long totalCount = 0L;
            for (Backet backet: backets) {
                totalCount += backet.preciseCount;
                // For precise histogram, left bound is inclusive right bound
                statsConsumer.stat(
                    prefix + '-' + metricName
                    + '-' + backet.leftBound + "ms_ammm",
                    totalCount);
            }
        }
        statsConsumer.stat(
            StringUtils.concat(prefix, '-', metricName, '-', "hist_ahhh"),
            hist);
    }

    @Override
    public List<GolovanChart> createGolovanCharts(
        final ImmutableGolovanPanelConfig config,
        final String name)
    {
        GolovanChart chart = new GolovanChart(
            namingProvider.chartIdSuffix(),
            namingProvider.chartNameSuffix(),
            false,
            false,
            0d);
        List<String> colors =
            GolovanPanel.colors(GolovanPanel.ALL_COLORS, backets.length);
        for (int i = backets.length; i-- > 0;) {
            long leftBound = backets[i].leftBound;
            chart.addSignal(
                new GolovanSignal(
                    "hperc(" + name + "-times-hist_ahhh,0,"
                    + backets[i].leftBound + ')',
                    config.tag(),
                    namingProvider.signalName(leftBound),
                    colors.get(i),
                    1,
                    true));
        }
        return Collections.singletonList(chart);
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("RequestTimeHistogramMetric[prefix=");
        sb.append(prefix);
        sb.append(" namingSuffix=");
        sb.append(namingProvider.chartIdSuffix());
        sb.append(" preciseHistogram=");
        sb.append(preciseHistogram);
        sb.append(" processingTimeStats=");
        sb.append(processingTimeStats);
        sb.append(" backets");
        char sep = '=';
        for (Backet backet : backets) {
            sb.append(sep);
            sb.append(backet);
            sep = ',';
        }
        sb.append(" processingBackets");
        sep = '=';
        for (Backet backet : processingBackets) {
            sb.append(sep);
            sb.append(backet);
            sep = ',';
        }
        sb.append(']');
        return new String(sb);
    }

    private static class Backet implements Comparable<Backet> {
        private final long leftBound;
        private long count = 0L;
        private long preciseCount = 0L;

        Backet(final long leftBound) {
            this.leftBound = leftBound;
        }

        @Override
        public int compareTo(final Backet other) {
            return Long.compare(leftBound, other.leftBound);
        }

        @Override
        public String toString() {
            final StringBuilder sb = new StringBuilder();
            sb.append("<").append(leftBound).append(">");
            return new String(sb);
        }
    }

    public interface NamingProvider {
        String chartIdSuffix();

        String chartNameSuffix();

        String signalName(long value);
    }

    public enum DefaultNamingProvider implements NamingProvider {
        INSTANCE;

        @Override
        public String chartIdSuffix() {
            return "-timings";
        }

        @Override
        public String chartNameSuffix() {
            return " Timings (%)";
        }

        @Override
        public String signalName(final long value) {
            if (value < 1000L) {
                return value + "ms";
            } else {
                return (value / 1000d) + "s";
            }
        }
    }

    public static class BasicNamingProvider implements NamingProvider {
        private final String chartIdSuffix;
        private final String chartNameSuffix;
        private final double valueDivisor;

        public BasicNamingProvider(
            final String chartIdSuffix,
            final String chartNameSuffix,
            final double valueDivisor)
        {
            this.chartIdSuffix = chartIdSuffix;
            this.chartNameSuffix = chartNameSuffix;
            this.valueDivisor = valueDivisor;
        }

        @Override
        public String chartIdSuffix() {
            return chartIdSuffix;
        }

        @Override
        public String chartNameSuffix() {
            return chartNameSuffix;
        }

        @Override
        public String signalName(final long value) {
            return Double.toString(value / valueDivisor);
        }
    }
}

