package ru.yandex.stater;

import java.util.Collections;
import java.util.List;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.util.string.StringUtils;

public class RequestTimeMetric
    implements Metric, MetricBuilder, PassiveStater<RequestInfo>
{
    private final String prefix;
    private final boolean processingTimeStats;
    private long maxTime;
    private long maxProcessingTime;
    private long totalTime;
    private long totalProcessingTime;

    public RequestTimeMetric(final IniConfig config) throws ConfigException {
        prefix = "";
        processingTimeStats = config.getBoolean("processing-time-stats", true);
    }

    public RequestTimeMetric(
        final String prefix,
        final boolean processingTimeStats)
    {
        this.prefix = prefix;
        this.processingTimeStats = processingTimeStats;
    }

    @Override
    public RequestTimeMetric build(final String prefix) {
        return new RequestTimeMetric(prefix, processingTimeStats);
    }

    @Override
    public void accept(final RequestInfo info) {
        long requestTime = info.requestTime();
        totalTime += requestTime;
        if (requestTime > maxTime) {
            maxTime = requestTime;
        }
        if (processingTimeStats) {
            long processingTime = info.processingTime();
            totalProcessingTime += processingTime;
            if (processingTime > maxProcessingTime) {
                maxProcessingTime = processingTime;
            }
        }
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer,
        final int timeDiff)
        throws E
    {
        stats(statsConsumer);
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        statsConsumer.stat(
            StringUtils.concat(prefix, "-max-time_axxx"),
            maxTime);
        statsConsumer.stat(
            StringUtils.concat(prefix, "-total-time_ammm"),
            totalTime);
        if (processingTimeStats) {
            statsConsumer.stat(
                StringUtils.concat(prefix, "-processing-max-time_axxx"),
                maxProcessingTime);
            statsConsumer.stat(
                StringUtils.concat(prefix, "-processing-total-time_ammm"),
                totalProcessingTime);
        }
    }

    @Override
    public List<GolovanChart> createGolovanCharts(
        final ImmutableGolovanPanelConfig config,
        final String name)
    {
        GolovanChart chart = new GolovanChart(
            "-mean-time",
            " mean time (ms)",
            false,
            false,
            0d);
        chart.addSplitSignal(
            config,
            "div(" + name + "-total-time_ammm,"
            + name + "-total_ammm)",
            1,
            true,
            false);
        return Collections.singletonList(chart);
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("RequestTimeMetric[prefix=");
        sb.append(prefix);
        sb.append(" processingTimeStats=");
        sb.append(processingTimeStats);
        sb.append(']');
        return new String(sb);
    }
}
