package ru.yandex.stater;

import java.util.List;

import javax.annotation.Nonnull;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.util.string.StringUtils;

public class RequestsStater
    extends TimeFrameQueue<RequestInfo>
    implements Stater
{
    private final int metricsTimeFrame;
    private final String prefix;
    private final List<MetricBuilder> metricBuilders;
    private final String category;
    private final String title;

    public RequestsStater(final ImmutableStaterConfig config) {
        this(
            config.metricsTimeFrame(),
            config.prefix(),
            config.metrics(),
            config.category(),
            config.title());
    }

    public RequestsStater(
        final long metricsTimeFrame,
        final String prefix,
        final List<MetricBuilder> metricBuilders,
        final String category,
        final String title)
    {
        super(metricsTimeFrame);
        this.metricsTimeFrame = (int) metricsTimeFrame;
        this.prefix = prefix;
        this.metricBuilders = metricBuilders;
        this.category = category;
        this.title = title;
    }

    public String prefix() {
        return prefix;
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        Metric[] metrics = new Metric[metricBuilders.size()];
        for (int i = 0; i < metrics.length; i++) {
            metrics[i] = metricBuilders.get(i).build(prefix);
        }
        for (RequestInfo info: this) {
            for (Metric metric : metrics) {
                metric.accept(info);
            }
        }
        for (Metric metric: metrics) {
            metric.stats(statsConsumer, metricsTimeFrame);
        }
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("RequestStater[prefix=");
        sb.append(prefix);
        sb.append(" metrics");
        char sep = '=';
        for (MetricBuilder metric: metricBuilders) {
            sb.append(sep);
            sb.append(metric);
            sep = ',';
        }
        sb.append(']');
        return new String(sb);
    }

    @Override
    public void addToGolovanPanel(
        @Nonnull final GolovanPanel panel,
        @Nonnull final String statsPrefix)
    {
        if (prefix.startsWith("ignore")) {
            return;
        }

        ImmutableGolovanPanelConfig config = panel.config();
        String chartsPrefix =
            statsPrefix + StringUtils.removeSuffix(prefix, '-');
        GolovanChartGroup group =
            new GolovanChartGroup(chartsPrefix, statsPrefix);
        for (MetricBuilder builder: metricBuilders) {
            for (GolovanChart chart
                : builder.createGolovanCharts(config, chartsPrefix))
            {
                group.addChart(chart);
            }
        }
        panel.addCharts(category, title, group);
    }

    @Override
    public void addToAlertsConfig(
        final IniConfig alertsConfig,
        final ImmutableGolovanPanelConfig panelConfig,
        final String statsPrefix)
        throws BadRequestException
    {
        if (prefix.startsWith("ignore")) {
            return;
        }

        String chartsPrefix =
            statsPrefix + StringUtils.removeSuffix(prefix, '-');
        for (MetricBuilder builder: metricBuilders) {
            builder.addAlerts(alertsConfig, panelConfig, chartsPrefix);
        }
    }
}

