package ru.yandex.stater;

import java.util.Arrays;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.util.string.StringUtils;

public class SimpleHistogramMetric implements PassiveStater<Long> {
    private static final String INF = "inf";
    private static final String DEFAULT_TOPS =
        "0, 1, 10, 50, 100, 150, 200, 300, 400, 500, 600, 700, 800, 900, 1000,"
            + " 1500, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000,"
            + " 20000, 30000, 100000";
    private final String prefix;
    private final Backet[] backets;
    private final boolean preciseHistogram;

    public SimpleHistogramMetric(final IniConfig config)
        throws ConfigException
    {
        prefix = config.getString("prefix", "");
        preciseHistogram = config.getBoolean("precise-histogram", true);
        final String backetsString =
            config.getString(
                "histogram-ranges",
                DEFAULT_TOPS);
        if (backetsString.trim().length() == 0) {
            backets = new Backet[0];
        } else {
            String[] backets = backetsString.split("[,;]");
            this.backets = new Backet[backets.length];
            for (int i = 0; i < backets.length; ++i) {
                String backet = backets[i];
                final long leftBound;
                if (backet.trim().equalsIgnoreCase(INF)
                    || backet.trim().equalsIgnoreCase("max"))
                {
                    leftBound = Long.MAX_VALUE;
                } else {
                    leftBound = Long.parseLong(backet.trim());
                }
                this.backets[i] = new Backet(leftBound);
            }
            Arrays.sort(backets);
        }
    }

    public SimpleHistogramMetric(
        final SimpleHistogramMetric other)
    {
        this(other, other.prefix);
    }

    public SimpleHistogramMetric(
        final SimpleHistogramMetric other,
        final String prefix)
    {
        this.prefix = prefix;
        backets = copy(other.backets);
        preciseHistogram = other.preciseHistogram;
    }

    private static Backet[] copy(final Backet[] example) {
        Backet[] copy = new Backet[example.length];
        for (int i = 0; i < example.length; ++i) {
            copy[i] = new Backet(example[i].leftBound);
        }
        return copy;
    }

    @Override
    public void accept(final Long time) {
        collectTimes(backets, time);
    }

    private void collectTimes(
        final Backet[] backets,
        final long time)
    {
        int len = backets.length;
        // left bound is exclusive for all histogram backets
        // except zero time backet
        // for precise histogram, left bound actually is inclusive right bound
        int i = 0;
        for (; i < len; ++i) {
            Backet backet = backets[i];
            if (time <= backet.leftBound) {
                ++backet.preciseCount;
                break;
            }
        }
        if (i == 0) {
            ++backets[0].count;
        } else {
            ++backets[i - 1].count;
        }
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        backets(statsConsumer, backets, "times");
    }

    private <E extends Exception> void backets(
        final StatsConsumer<? extends E> statsConsumer,
        final Backet[] backets,
        final String metricName)
        throws E
    {
        if (backets.length == 0) {
            return;
        }
        long[] hist = new long[backets.length << 1];
        int pos = 0;
        for (Backet backet: backets) {
            hist[pos++] = backet.leftBound;
            hist[pos++] = backet.count;
        }
        if (preciseHistogram) {
            long totalCount = 0L;
            for (Backet backet: backets) {
                totalCount += backet.preciseCount;
                // For precise histogram, left bound is inclusive right bound
                statsConsumer.stat(
                    prefix + '-' + metricName
                        + '-' + backet.leftBound + "ms_ammm",
                    totalCount);
            }
        }

        statsConsumer.stat(
            StringUtils.concat(prefix, '-', metricName, '-', "hist_ahhh"),
            hist);
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("SimpleHistogramMetric[prefix=");
        sb.append(prefix);
        sb.append(" preciseHistogram=");
        sb.append(preciseHistogram);
        sb.append(" backets=");
        String sep = "";
        for (Backet backet : backets) {
            sb.append(sep);
            sb.append(backet);
            sep = ",";
        }
        sb.append(']');
        return new String(sb);
    }

    private static class Backet implements Comparable<Backet> {
        private final long leftBound;
        private long count = 0L;
        private long preciseCount = 0L;

        Backet(final long leftBound) {
            this.leftBound = leftBound;
        }

        @Override
        public int compareTo(final Backet other) {
            return Long.compare(leftBound, other.leftBound);
        }

        @Override
        public String toString() {
            final StringBuilder sb = new StringBuilder();
            sb.append("<").append(leftBound).append(">");
            return new String(sb);
        }
    }
}
