package ru.yandex.http.tskv;

import java.util.Objects;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.nio.protocol.HttpAsyncResponseProducer;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.http.proxy.FilterProxySession;
import ru.yandex.http.proxy.PrefixedProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.http.util.server.SessionContext;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.tskv.BasicPrefixedTskvLogger;
import ru.yandex.tskv.TskvString;
import ru.yandex.util.timesource.TimeSource;

public class TskvProxySession extends FilterProxySession {
    private static final DateTimeFormatter TIMESTAMP_FORMATTER =
        DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZZ")
            .withZone(DateTimeZone.forID("Europe/Moscow"));

    private final long sessionStartTimestamp;
    private final BasicPrefixedTskvLogger tskvLogger;

    public TskvProxySession(
        final ProxySession session,
        final TskvLoggerProvider tskvLoggerProvider,
        final HttpAsyncRequestHandler<?> handler,
        final String initMessage,
        final String... commonParams)
        throws HttpException
    {
        super(session);
        sessionStartTimestamp = TimeSource.INSTANCE.currentTimeMillis();
        if ((commonParams.length & 1) != 0) {
            throw new IllegalArgumentException("Odd number of common params");
        }
        BasicPrefixedTskvLogger tskvLogger = tskvLoggerProvider.tskvLogger();
        TskvString tskvPrefix =
            tskvLogger.record()
                .append(
                    CommonTskvFields.HANDLER,
                    handler.getClass().getSimpleName())
                .append(CommonTskvFields.HOSTNAME, SessionContext.HOSTNAME)
                .append(
                    CommonTskvFields.SESSION_START_TIMESTAMP_MS,
                    TimeSource.INSTANCE.currentTimeMillis())
                .append(
                    CommonTskvFields.TVM_SRC_ID,
                    Objects.toString(
                        context().getAttribute(HttpServer.TVM_SRC_ID)))
                .append(
                    CommonTskvFields.TVM_USER_UID,
                    Objects.toString(
                        context().getAttribute(HttpServer.TVM_USER_UID)))
                .append(
                    CommonTskvFields.SESSION_ID,
                    Objects.toString(
                        context().getAttribute(HttpServer.SESSION_ID)))
                .append(
                    CommonTskvFields.X_REQUEST_ID,
                    headers().getString(YandexHeaders.X_REQUEST_ID, ""))
                .append(
                    CommonTskvFields.REQUEST_URI,
                    request().getRequestLine().getUri())
                .append(
                    CommonTskvFields.REMOTE_IP,
                    Objects.toString(connection().getRemoteAddress()));

        for (int i = 0; i < commonParams.length; i += 2) {
            tskvPrefix =
                tskvPrefix.append(commonParams[i], commonParams[i + 1]);
        }

        this.tskvLogger = tskvLogger.prefix(tskvPrefix);
        if (initMessage != null) {
            tskvLog(CommonTskvFields.Stage.INITIAL, initMessage);
        }
    }

    private TskvProxySession(
        final TskvProxySession sample,
        final String key,
        final String value)
    {
        super(new PrefixedProxySession(sample.session, value));
        sessionStartTimestamp = sample.sessionStartTimestamp;
        tskvLogger =
            sample.tskvLogger.prefix(
                sample.tskvLogger.record().append(key, value));
    }

    public TskvProxySession addPrefix(final String key, final String value) {
        return new TskvProxySession(this, key, value);
    }

    public void tskvLog(final String message) {
        tskvLog(CommonTskvFields.Stage.INTERMEDIATE, message, "");
    }

    public void tskvLog(
        final CommonTskvFields.Stage stage,
        final String message)
    {
        tskvLog(stage, message, "");
    }

    public void tskvLog(
        final CommonTskvFields.Stage stage,
        final String message,
        final String status)
    {
        long timestamp = TimeSource.INSTANCE.currentTimeMillis();
        tskvLogger.log(
            tskvLogger.record()
                .append(CommonTskvFields.UNIXTIME, timestamp / 1000L)
                .append(CommonTskvFields.TIMESTAMP_MS, timestamp)
                    .append(
                        CommonTskvFields.SESSION_LENGTH_MS,
                        timestamp - sessionStartTimestamp)
                .append(CommonTskvFields.STAGE, stage.toString())
                .append(
                    CommonTskvFields.TIMESTAMP,
                    TIMESTAMP_FORMATTER.print(timestamp))
                .append(CommonTskvFields.MESSAGE, message)
                .append(CommonTskvFields.STATUS, status));
    }

    @Override
    public void response(final int status) {
        super.response(status);
        tskvLog(
            CommonTskvFields.Stage.COMPLETED,
            "Request completed",
            Integer.toString(status));
    }

    @Override
    public void response(final int status, final String body) {
        super.response(status, body);
        tskvLog(
            CommonTskvFields.Stage.COMPLETED,
            "Request completed",
            Integer.toString(status));
    }

    @Override
    public void response(final int status, final HttpEntity entity) {
        super.response(status, entity);
        tskvLog(
            CommonTskvFields.Stage.COMPLETED,
            "Request completed",
            Integer.toString(status));
    }

    @Override
    public void response(final HttpResponse response) {
        super.response(response);
        tskvLog(
            CommonTskvFields.Stage.COMPLETED,
            "Request completed",
            Integer.toString(response.getStatusLine().getStatusCode()));
    }

    @Override
    public void response(final HttpAsyncResponseProducer producer) {
        super.response(producer);
        tskvLog(
            CommonTskvFields.Stage.COMPLETED,
            "Request completed",
            Integer.toString(
                producer.generateResponse().getStatusLine().getStatusCode()));
    }

    @Override
    public void handleException(final HttpException e) {
        super.handleException(e);
        String status;
        if (e instanceof ServerException) {
            status = Integer.toString(((ServerException) e).statusCode());
        } else {
            status = e.getClass().getName();
        }
        StringBuilderWriter sbw = new StringBuilderWriter();
        sbw.append("Request failed: ");
        e.printStackTrace(sbw);
        tskvLog(CommonTskvFields.Stage.FAILED, sbw.toString(), status);
    }
}

