package ru.yandex.client.tvm2;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpMessage;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;

import ru.yandex.concurrent.CloseableTimerTask;
import ru.yandex.concurrent.TimerCloser;
import ru.yandex.function.SimpleGenericAutoCloseableHolder;
import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.io.GenericCloseableAdapter;
import ru.yandex.json.parser.JsonException;
import ru.yandex.passport.tvmauth.BlackboxEnv;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;
import ru.yandex.passport.tvmauth.CheckedUserTicket;
import ru.yandex.passport.tvmauth.Utils;
import ru.yandex.passport.tvmauth.Version;
import ru.yandex.passport.tvmauth.deprecated.ServiceContext;
import ru.yandex.passport.tvmauth.deprecated.UserContext;

public class Tvm2ServiceContextRenewalTask extends CloseableTimerTask {
    public enum Status {
        OK,
        EXPIRING,
        EXPIRED
    }

    private static final long WARNING_LAG = 3600L * 1000L * 3L;
    private static final long CRITICAL_LAG = 3600L * 1000L * 11L;

    private final Logger logger;
    private final ImmutableTvm2ServiceConfig config;
    private final Timer timer;
    private final BlackboxEnv blackboxEnv;
    private final int clientId;
    private final String secret;
    private final long keysRenewalInterval;
    private final CloseableHttpClient client;
    private final URI keysUri;
    private volatile Context context;

    public Tvm2ServiceContextRenewalTask(
        final Logger logger,
        final ImmutableTvm2ServiceConfig config,
        final ImmutableDnsConfig dnsConfig)
        throws HttpException, IOException, JsonException, URISyntaxException
    {
        try (SimpleGenericAutoCloseableHolder<IOException> holder =
                new SimpleGenericAutoCloseableHolder<>(closeChain))
        {
            this.logger = logger;
            this.config = config;
            timer = new Timer("Tvm2Renewal", true);
            closeChain.add(new TimerCloser(timer));
            blackboxEnv = config.blackboxEnv();
            clientId = config.clientId();
            secret = config.secret();
            keysRenewalInterval = config.keysRenewalInterval();
            client = ClientBuilder.createClient(config, dnsConfig);
            closeChain.add(new GenericCloseableAdapter<>(client));
            HttpHost host = config.host();
            keysUri =
                new URI(
                    host.getSchemeName(),
                    null,
                    host.getHostName(),
                    host.getPort(),
                    "/2/keys/",
                    "lib_version=" + Version.get(),
                    null)
                    .parseServerAuthority();
            context = createContext();
            holder.release();
        }
    }

    public ImmutableTvm2ServiceConfig config() {
        return config;
    }

    public CloseableHttpClient client() {
        return client;
    }

    public ServiceContext serviceContext() {
        return context.serviceContext;
    }

    public void schedule(final TimerTask task, final long interval) {
        timer.schedule(task, interval, interval);
    }

    public void start() {
        schedule(this, keysRenewalInterval);
    }

    private static Header removeSignature(final Header header) {
        return HeaderUtils.createHeader(
            header.getName(),
            Utils.removeTicketSignature(header.getValue()));
    }

    public AuthResult checkAuthorization(
        final HttpMessage message,
        final String headerName,
        final Set<Integer> allowedSrcs)
    {
        Header header = message.getFirstHeader(headerName);
        if (header == null) {
            return new AuthResult(headerName + " header is not set");
        }
        String value = header.getValue();
        if (value == null) {
            return new AuthResult(headerName + " header value is null");
        }
        ServiceContext serviceContext = context.serviceContext;
        if (serviceContext == null) {
            return new AuthResult("No service context defined");
        }
        CheckedServiceTicket ticket;
        try {
            ticket = serviceContext.check(value);
        } catch (Throwable t) {
            return new AuthResult("Service ticket check failed: " + t);
        }
        if (!ticket.booleanValue()) {
            return new AuthResult(
                "Invalid header: " + removeSignature(header)
                + ", error: " + ticket.getStatus());
        }
        int src = ticket.getSrc();
        if (!allowedSrcs.contains(src)) {
            return new AuthResult(
                "Ticket " + removeSignature(header)
                + " was issued by client " + src
                + " which doesn't present in allowed sources: " + allowedSrcs);
        }
        return new AuthResult(ticket);
    }

    public UserAuthResult checkUserAuthorization(
        final HttpMessage message,
        final String headerName)
    {
        return checkUserAuthorization(message, headerName, true);
    }

    // If ticket is not required, return null instead of error when ticket is
    // absent
    public UserAuthResult checkUserAuthorization(
        final HttpMessage message,
        final String headerName,
        final boolean requireTicket)
    {
        Header header = message.getFirstHeader(headerName);
        if (header == null) {
            if (requireTicket) {
                return new UserAuthResult(headerName + " header is not set");
            } else {
                return null;
            }
        }
        String value = header.getValue();
        if (value == null) {
            if (requireTicket) {
                return new UserAuthResult(
                    headerName + " header value is null");
            } else {
                return null;
            }
        }
        CheckedUserTicket ticket;
        try {
            ticket = context.userContext.check(value);
        } catch (Throwable t) {
            return new UserAuthResult("User ticket check failed: " + t);
        }
        if (!ticket.booleanValue()) {
            return new UserAuthResult(
                "Invalid header: " + removeSignature(header)
                + ", error: " + ticket.getStatus());
        }
        return new UserAuthResult(ticket);
    }

    private Context createContext() throws HttpException, IOException {
        logger.info("Requesting keys from URI: " + keysUri.toString());
        ServiceContext serviceContext;
        UserContext userContext;
        try (CloseableHttpResponse response = client.execute(
                new HttpGet(keysUri)))
        {
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(keysUri, response);
            }
            String keys = CharsetUtils.toString(response.getEntity());
            logger.info("Keys received");
            userContext = new UserContext(blackboxEnv, keys);
            if (secret == null) {
                serviceContext = null;
            } else {
                serviceContext = new ServiceContext(clientId, secret, keys);
                logger.info("Service context created");
            }
        }
        return new Context(serviceContext, userContext);
    }

    @Override
    public void run() {
        try {
            context = createContext();
        } catch (Throwable t) {
            logger.log(Level.WARNING, "ServiceContext renewal failed", t);
        }
    }

    public Status status() {
        long lag = System.currentTimeMillis() - context.timestamp;
        if (lag < WARNING_LAG) {
            return Status.OK;
        } else if (lag < CRITICAL_LAG) {
            return Status.EXPIRING;
        } else {
            return Status.EXPIRED;
        }
    }

    private static class Context {
        private final long timestamp = System.currentTimeMillis();
        private final ServiceContext serviceContext;
        private final UserContext userContext;

        Context(
            final ServiceContext serviceContext,
            final UserContext userContext)
        {
            this.serviceContext = serviceContext;
            this.userContext = userContext;
        }
    }
}

