package ru.yandex.client.tvm2;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;

import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.util.timesource.TimeSource;

public class Tvm2TicketRenewalTask extends TimerTask implements Tvm2TicketSupplier {
    private final Tvm2ServiceContextRenewalTask serviceContextRenewalTask;
    private final Logger logger;
    private final String destinationClientId;
    private final long renewalInterval;
    private final URI uri;
    private volatile Map<String, String> tickets;

    public Tvm2TicketRenewalTask(
        final Logger logger,
        final Tvm2ServiceContextRenewalTask serviceContextRenewalTask,
        final ImmutableTvm2ClientConfig config)
        throws HttpException, IOException, JsonException, URISyntaxException
    {
        this.logger = logger;
        this.serviceContextRenewalTask = serviceContextRenewalTask;
        destinationClientId = config.destinationClientId();
        renewalInterval = config.renewalInterval();
        HttpHost host = serviceContextRenewalTask.config().host();
        uri =
            new URI(
                host.getSchemeName(),
                null,
                host.getHostName(),
                host.getPort(),
                "/2/ticket/",
                null,
                null)
                .parseServerAuthority();
        tickets = requestTickets();
    }

    public void start() {
        serviceContextRenewalTask.schedule(this, renewalInterval);
    }

    @Override
    public String ticket() {
        return tickets.values().iterator().next();
    }

    @Override
    public String ticket(final String destinationClientId) {
        return tickets.get(destinationClientId);
    }

    private Map<String, String> requestTickets()
        throws HttpException, IOException, JsonException
    {
        long now = TimeSource.INSTANCE.currentTimeMillis();
        String ts = Long.toString(TimeUnit.MILLISECONDS.toSeconds(now));
        StringBuilder sb =
            new StringBuilder("grant_type=client_credentials&src=");
        sb.append(serviceContextRenewalTask.config().clientId());
        sb.append("&dst=");
        sb.append(destinationClientId);
        sb.append("&ts=");
        sb.append(ts);
        sb.append("&sign=");
        sb.append(
            serviceContextRenewalTask.serviceContext()
                .signCgiParamsForTvm(ts, destinationClientId));
        String request = new String(sb);
        logger.info("Requesting tickets");
        HttpPost post = new HttpPost(uri);
        post.setEntity(
            new StringEntity(
                request,
                ContentType.APPLICATION_FORM_URLENCODED));
        try (CloseableHttpResponse response =
                serviceContextRenewalTask.client().execute(post))
        {
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response);
            }
            String body = CharsetUtils.toString(response.getEntity());
            Map<String, String> tickets = new LinkedHashMap<>();
            try {
                JsonMap root = TypesafeValueContentHandler.parse(body).asMap();
                for (Map.Entry<String, JsonObject> entry: root.entrySet()) {
                    tickets.put(
                        entry.getKey(),
                        entry.getValue().get("ticket").asString());
                }
            } catch (Throwable t) {
                logger.log(
                    Level.WARNING,
                    "Failed to extract tickets from " + body,
                    t);
                throw t;
            }
            if (tickets.isEmpty()) {
                throw new JsonException("No tickets extracted from " + body);
            }
            logger.info("Tickets received for clients: " + tickets.keySet());
            return tickets;
        } catch (Throwable t) {
            logger.log(
                Level.WARNING,
                "Failed to receive tickets for " + request,
                t);
            throw t;
        }
    }

    @Override
    public void run() {
        try {
            tickets = requestTickets();
        } catch (Throwable t) {
            logger.log(Level.WARNING, "TVM2 tickets renewal failed", t);
        }
    }
}

