package ru.yandex.parser.uri;

import java.io.File;
import java.math.BigInteger;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpRequest;
import org.apache.http.util.EntityUtils;

import ru.yandex.collection.Iterators;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.util.BadRequestException;

public class OpenApiCgiParams extends CgiParams {
    private static final long serialVersionUID = 0L;

    private final OpenApiPath path;
    private boolean track = true;

    public OpenApiCgiParams(final HttpRequest request, final QueryParser parser) {
        super(parser);

        String method = request.getRequestLine().getMethod();
        OpenApiPath path =
            new OpenApiPath(
                method,
                new UriParser(request.getRequestLine().getUri()).path().decodeOrRaw());
        OpenApiPath curPath = OpenApiDocumentation.paths().putIfAbsent(path, path);
        if (curPath == null) {
            this.path = path;
        } else {
            this.path = curPath;
        }

        if (request instanceof HttpEntityEnclosingRequest) {
            try {
                String body = EntityUtils.toString(
                    ((HttpEntityEnclosingRequest) request).getEntity(),
                    StandardCharsets.UTF_8);
                OpenApiParameter parameter = new OpenApiParameter();
                parameter.area(OpenApiParameter.Area.BODY).name("body").examples().add(body);
                this.path.param("body", parameter);
            } catch (Exception e) {
                System.err.println("Failed to get request body for documentation");
            }
        }
    }

    public OpenApiPath path() {
        return path;
    }

    public void response(final int code, final String body) {
        this.path.response(code, body);
    }

    private OpenApiParameter parameter(
        final String name,
        final OpenApiParameter.Type type,
        final Object defaultValue,
        final String description,
        final boolean required)
    {
        OpenApiParameter parameter = new OpenApiParameter();
        parameter.name(name).type(type).required(required).description(description);
        if (defaultValue != null) {
            parameter.defaultValue(defaultValue);
        }

        this.path.param(name, parameter);
        return parameter;
    }

    private OpenApiParameter addStringParameter(
        final String name,
        final String value,
        final String defaultValue,
        final String description,
        final boolean required)
    {
        OpenApiParameter parameter =
            parameter(name, OpenApiParameter.Type.STRING, defaultValue, description, required);
        parameter.examples().add(value);

        return parameter;
    }

    private synchronized OpenApiParameter addStringArrayParameter(
        final String name,
        final List<String> value,
        final List<String> defaultValue,
        final String description,
        final boolean required)
    {
        OpenApiParameter parameter =
            parameter(name, OpenApiParameter.Type.ARRAY, defaultValue, description, required);

        if (value != null) {
            parameter.examples().add(String.join(",", value));
        }
        return parameter;
    }

    @Override
    public synchronized String getOrNull(final String name) {
        if (!track) {
            return super.getOrNull(name);
        }

        track = false;
        String value = super.getOrNull(name);
        track = true;
        addStringParameter(name, value, null, "", false);
        return value;
    }

    @Override
    public String getLastOrNull(final String name) {
        if (!track) {
            return super.getLastOrNull(name);
        }

        track = false;
        String value = super.getLastOrNull(name);
        track = true;
        addStringParameter(name, value, null, "", false);
        return value;
    }

    @Override
    public Iterator<String> getAllOrNull(final String name) {
        if (!track) {
            return super.getAllOrNull(name);
        }

        track = false;
        Iterator<String> valueIt = super.getAllOrNull(name);
        if (valueIt != null) {
            addStringArrayParameter(name, Iterators.toList(super.getAllOrNull(name)), null, "", false);
        } else {
            addStringArrayParameter(name, null, null, "", false);
        }
        track = true;
        return valueIt;
    }

    @Override
    public List<String> getAll(final String name) {
        if (!track) {
            return super.getAll(name);
        }

        track = false;
        List<String> value = super.getAll(name);
        track = true;
        addStringArrayParameter(name, value, null, "", true);
        return value;
    }

    @Override
    public <T> T get(final String name, final GenericFunction<String, ? extends T, ? extends Exception> parser) throws BadRequestException {
        if (!track) {
            return super.get(name, parser);
        }

        track = false;
        T res = super.get(name, parser);

        addStringParameter(name, getOrNull(name), null, "", true).parser(parser);
        track = true;
        return res;
    }

    @Override
    public <T> T get(final String name, final T defaultValue, final GenericFunction<String, ? extends T, ? extends Exception> parser) throws BadRequestException {
        if (!track) {
            return super.get(name, defaultValue, parser);
        }

        track = false;
        T res = super.get(name, defaultValue, parser);

        addStringParameter(
            name,
            getOrNull(name),
            String.valueOf(defaultValue),
            "",
            false)
            .parser(parser);
        track = true;
        return res;
    }

    @Override
    public <T> T getLast(final String name, final GenericFunction<String, ? extends T, ? extends Exception> parser) throws BadRequestException {
        return super.getLast(name, parser);
    }

    @Override
    public <T> T getLast(final String name, final T defaultValue, final GenericFunction<String, ? extends T, ? extends Exception> parser) throws BadRequestException {
        return super.getLast(name, defaultValue, parser);
    }

    @Override
    public <T, C extends Collection<? super T>> C getAll(
        final String name,
        final GenericFunction<String, ? extends T, ? extends Exception> parser,
        final C collection)
        throws BadRequestException
    {
        if (!track) {
            return super.getAll(name, parser, collection);
        }

        track = false;
        C value = super.getAll(name, parser, collection);

        List<String> valueStr = value.stream().map(String::valueOf).collect(Collectors.toList());
        addStringArrayParameter(name, valueStr, null, "Parsed by " + parser,  true);
        track = true;
        return value;
    }

    @Override
    public <T, C extends Collection<? super T>> C getAll(
        final String name,
        final C defaultValue,
        final GenericFunction<String, ? extends T, ? extends Exception> parser,
        final C collection)
        throws BadRequestException
    {
        if (!track) {
            return super.getAll(name, defaultValue, parser, collection);
        }

        track = false;
        C value = super.getAll(name, defaultValue, parser, collection);
        List<String> valueStr;
        if (value != null) {
            valueStr = value.stream().map(String::valueOf).collect(Collectors.toList());
        } else {
            valueStr = null;
        }
        addStringArrayParameter(
            name,
            valueStr,
            null,
            "",
            true)
            .parser(parser);
        track = true;
        return value;
    }

    @Override
    public <T, C extends Collection<T>> C getAll(
        final String name,
        final GenericFunction<String, ? extends C, ? extends Exception> parser)
        throws BadRequestException
    {
        if (!track) {
            return super.getAll(name, parser);
        }

        track = false;
        C value = super.getAll(name, parser);
        List<String> valueStr;
        if (value != null) {
            valueStr = value.stream().map(String::valueOf).collect(Collectors.toList());
        } else {
            valueStr = null;
        }
        addStringArrayParameter(
            name,
            valueStr,
            null,
            "",
            true).parser(parser);
        track = true;
        return value;
    }

    @Override
    public <T, C extends Collection<T>> C getAll(final String name, final C defaultValue, final GenericFunction<String, ? extends C, ? extends Exception> parser) throws BadRequestException {
        if (!track) {
            return super.getAll(name, defaultValue, parser);
        }

        track = false;
        C value = super.getAll(name, defaultValue, parser);
        List<String> valueStr;
        if (value != null) {
            valueStr = value.stream().map(String::valueOf).collect(Collectors.toList());
        } else {
            valueStr = null;
        }

        addStringArrayParameter(
            name,
            valueStr,
            null,
            "",
            false).parser(parser);

        track = true;
        return value;
    }

    @Override
    public String getString(final String name) throws BadRequestException {
        if (!track) {
            return super.getString(name);
        }

        track = false;
        String res = super.getString(name);

        if (res != null) {
            addStringParameter(name, getOrNull(name), null, "", true);
        }
        track = true;
        return res;
    }

    @Override
    public String getString(final String name, final String defaultValue) {
        if (!track) {
            return super.getString(name, defaultValue);
        }

        track = false;
        String res = super.getString(name, defaultValue);
        track = true;

        addStringParameter(name, getOrNull(name), String.valueOf(defaultValue), "", false);
        return res;
    }

    @Override
    public int getInt(final String name) throws BadRequestException {
        if (!track) {
            return super.getInt(name);
        }

        track = false;
        int res = super.getInt(name);
        track = true;

        parameter(name, OpenApiParameter.Type.INTEGER, null, "", true)
            .examples().add(String.valueOf(res));

        return res;
    }

    @Override
    public Integer getInt(final String name, final Integer defaultValue) throws BadRequestException {
        if (!track) {
            return super.getInt(name, defaultValue);
        }

        track = false;
        Integer res = super.getInt(name, defaultValue);
        track = true;

        parameter(name, OpenApiParameter.Type.INTEGER, defaultValue, "", false)
            .examples().add(String.valueOf(res));

        return res;
    }

    @Override
    public long getLong(final String name) throws BadRequestException {
        if (!track) {
            return super.getLong(name);
        }

        track = false;
        long res = super.getLong(name);
        track = true;

        parameter(name, OpenApiParameter.Type.INTEGER, null, "int64", true)
            .examples().add(String.valueOf(res));

        return res;
    }

    @Override
    public Long getLong(final String name, final Long defaultValue) throws BadRequestException {
        if (!track) {
            return super.getLong(name, defaultValue);
        }

        track = false;
        Long res = super.getLong(name, defaultValue);
        track = true;

        parameter(name, OpenApiParameter.Type.INTEGER, defaultValue, "int64", false)
            .examples().add(String.valueOf(res));

        return res;
    }

    @Override
    public double getDouble(final String name) throws BadRequestException {
        if (!track) {
            return super.getDouble(name);
        }

        track = false;
        double res = super.getDouble(name);
        track = true;

        parameter(name, OpenApiParameter.Type.NUMBER, null, "double", true)
            .examples().add(String.valueOf(res));

        return res;
    }

    @Override
    public Double getDouble(final String name, final Double defaultValue) throws BadRequestException {
        if (!track) {
            return super.getDouble(name, defaultValue);
        }

        track = false;
        Double res = super.getDouble(name, defaultValue);
        track = true;

        parameter(name, OpenApiParameter.Type.NUMBER, defaultValue, "double", false)
            .examples().add(String.valueOf(res));

        return res;
    }

    @Override
    public BigInteger getBigInteger(final String name) throws BadRequestException {
        if (!track) {
            return super.getBigInteger(name);
        }

        track = false;
        BigInteger res = super.getBigInteger(name);
        track = true;

        parameter(name, OpenApiParameter.Type.INTEGER, null, "big integer", true)
            .examples().add(String.valueOf(res));

        return res;
    }

    @Override
    public BigInteger getBigInteger(final String name, final BigInteger defaultValue) throws BadRequestException {
        if (!track) {
            return super.getBigInteger(name, defaultValue);
        }

        track = false;
        BigInteger res = super.getBigInteger(name, defaultValue);
        track = true;

        parameter(name, OpenApiParameter.Type.INTEGER, defaultValue, "big integer", false)
            .examples().add(String.valueOf(res));

        return res;
    }

    @Override
    public boolean getBoolean(final String name) throws BadRequestException {
        if (!track) {
            return super.getBoolean(name);
        }

        track = false;
        boolean res = super.getBoolean(name);
        track = true;

        parameter(name, OpenApiParameter.Type.BOOLEAN, null, "true/false/1/0", true)
            .examples().add(String.valueOf(res));

        return res;
    }

    @Override
    public Boolean getBoolean(final String name, final Boolean defaultValue) throws BadRequestException {
        if (!track) {
            return super.getBoolean(name, defaultValue);
        }

        track = false;
        Boolean res = super.getBoolean(name, defaultValue);
        track = true;

        parameter(name, OpenApiParameter.Type.BOOLEAN, defaultValue, "true/false/1/0", false)
            .examples().add(String.valueOf(res));

        return res;
    }

    @Override
    public <T extends Enum<T>> T getEnum(final Class<T> t, final String name) throws BadRequestException {
        return super.getEnum(t, name);
    }

    @Override
    public <T extends Enum<T>> T getEnum(final Class<T> t, final String name, final T defaultValue) throws BadRequestException {
        return super.getEnum(t, name, defaultValue);
    }

    @Override
    public File getInputFile(final String name) throws BadRequestException {
        return super.getInputFile(name);
    }

    @Override
    public File getInputFile(final String name, final File defaultValue) throws BadRequestException {
        return super.getInputFile(name, defaultValue);
    }

    @Override
    public File getOutputFile(final String name) throws BadRequestException {
        return super.getOutputFile(name);
    }

    @Override
    public File getOutputFile(final String name, final File defaultValue) throws BadRequestException {
        return super.getOutputFile(name, defaultValue);
    }

    @Override
    public File getDir(final String name) throws BadRequestException {
        return super.getDir(name);
    }

    @Override
    public File getDir(final String name, final File defaultValue) throws BadRequestException {
        return super.getDir(name, defaultValue);
    }

    @Override
    public File getExistingDir(final String name) throws BadRequestException {
        return super.getExistingDir(name);
    }

    @Override
    public File getExistingDir(final String name, final File defaultValue) throws BadRequestException {
        return super.getExistingDir(name, defaultValue);
    }

    @Override
    public Charset getCharset(final String name) throws BadRequestException {
        return super.getCharset(name);
    }

    @Override
    public Charset getCharset(final String name, final Charset defaultValue) throws BadRequestException {
        return super.getCharset(name, defaultValue);
    }

    @Override
    public Locale getLocale(final String name) throws BadRequestException {
        return super.getLocale(name);
    }

    @Override
    public Locale getLocale(final String name, final Locale defaultValue) throws BadRequestException {
        return super.getLocale(name, defaultValue);
    }

    @Override
    public Level getLogLevel(final String name) throws BadRequestException {
        return super.getLogLevel(name);
    }

    @Override
    public Level getLogLevel(final String name, final Level defaultValue) throws BadRequestException {
        return super.getLogLevel(name, defaultValue);
    }

    @Override
    public URI getURI(final String name) throws BadRequestException {
        return super.getURI(name);
    }

    @Override
    public URI getURI(final String name, final URI defaultValue) throws BadRequestException {
        return super.getURI(name, defaultValue);
    }

    @Override
    public int getIntegerDuration(final String name) throws BadRequestException {
        return super.getIntegerDuration(name);
    }

    @Override
    public Integer getIntegerDuration(final String name, final Integer defaultValue) throws BadRequestException {
        return super.getIntegerDuration(name, defaultValue);
    }

    @Override
    public long getLongDuration(final String name) throws BadRequestException {
        return super.getLongDuration(name);
    }

    @Override
    public Long getLongDuration(final String name, final Long defaultValue) throws BadRequestException {
        return super.getLongDuration(name, defaultValue);
    }
}
