package ru.yandex.parser.uri;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;
import java.util.NoSuchElementException;

public class PathParser implements Iterable<PctEncodedString> {
    private final char[] path;
    private final int off;
    private final int end;
    private final PctDecoder decoder;

    public PathParser(final String path) {
        this(path, StandardCharsets.UTF_8);
    }

    public PathParser(final String path, final Charset charset) {
        this(path, new PctDecoder(false, charset));
    }

    public PathParser(final String path, final PctDecoder decoder) {
        this(path.toCharArray(), decoder);
    }

    PathParser(final char[] path, final PctDecoder decoder) {
        this(path, 0, path.length, decoder);
    }

    // CSOFF: ParameterNumber
    PathParser(
        final char[] path,
        final int off,
        final int len,
        final PctDecoder decoder)
    {
        this.path = path;
        this.off = off;
        this.end = off + len;
        this.decoder = decoder;
    }
    // CSON: ParameterNumber

    public PctDecoder decoder() {
        return decoder;
    }

    @Override
    public Iterator<PctEncodedString> iterator() {
        return new PathIterator();
    }

    private class PathIterator implements Iterator<PctEncodedString> {
        // points to the first non-/ character
        private int pos = off - 1;

        PathIterator() {
            incrementPos();
        }

        private void incrementPos() {
            ++pos;
            while (pos < end && path[pos] == '/') {
                ++pos;
            }
        }

        @Override
        public boolean hasNext() {
            return pos < end;
        }

        @Override
        public PctEncodedString next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            int start = pos++;
            while (pos < end && path[pos] != '/') {
                ++pos;
            }
            int end = pos;
            incrementPos();
            return new PctEncodedString(
                path,
                start,
                end - start,
                decoder);
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}

