package ru.yandex.parser.uri;

import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

import ru.yandex.charset.Decoder;
import ru.yandex.function.CharArrayProcessable;
import ru.yandex.function.CharArrayVoidProcessor;

public class PctDecoder
    extends CharArrayProcessable
    implements CharArrayVoidProcessor<CharacterCodingException>
{
    private static final byte[] EMPTY_BUFFER = new byte[0];
    private static final char[] EMPTY_CBUFFER = new char[0];
    private static final int PCT_ELEMENT_LENGTH = 3;
    private static final int HIGH_OFFSET = 4;
    private static final int[] HEX = {
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        0, 1, 2, 3, 4, 5, 6, 7, 8, 9, -1, -1, -1, -1, -1, -1,
        -1, 10, 11, 12, 13, 14, 15, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, 10, 11, 12, 13, 14, 15
    };

    private final Appender appender = new Appender();
    private final Decoder decoder;
    private final boolean decodePlus;

    @SuppressWarnings("HidingField")
    private byte[] buf = EMPTY_BUFFER;
    private char[] tmpcbuf = EMPTY_CBUFFER;

    public PctDecoder(final boolean decodePlus) {
        this(decodePlus, StandardCharsets.UTF_8);
    }

    public PctDecoder(final boolean decodePlus, final Charset charset) {
        this.decodePlus = decodePlus;
        decoder = new Decoder(charset);
    }

    private static int fromHex(final char c) throws CharacterCodingException {
        int result;
        if (c >= HEX.length) {
            result = -1;
        } else {
            result = HEX[c];
        }
        if (result == -1) {
            throw new MalformedPctException("Invalid pct character: " + c);
        }
        return result;
    }

    private int decodingPos(final String str) {
        if (decodePlus) {
            int idx = str.indexOf('+');
            if (idx == -1) {
                idx = str.indexOf('%');
            } else {
                int percent = str.indexOf('%');
                if (percent != -1 && percent < idx) {
                    idx = percent;
                }
            }
            return idx;
        } else {
            return str.indexOf('%');
        }
    }

    private int decodingPos(final char[] cbuf, final int off, final int len) {
        int end = off + len;
        if (decodePlus) {
            for (int i = off; i < end; ++i) {
                if (cbuf[i] == '+') {
                    int pos = i;
                    for (int j = off; j < i; ++j) {
                        if (cbuf[j] == '%') {
                            pos = j;
                            break;
                        }
                    }
                    return pos;
                }
            }
        }
        int pos = -1;
        for (int i = off; i < end; ++i) {
            if (cbuf[i] == '%') {
                pos = i;
                break;
            }
        }
        return pos;
    }

    public String decode(final String str) throws CharacterCodingException {
        int pos = decodingPos(str);
        if (pos == -1) {
            return str;
        } else {
            int len = str.length();
            if (tmpcbuf.length < len) {
                tmpcbuf = new char[Math.max(len, tmpcbuf.length << 1)];
            }
            str.getChars(0, len, tmpcbuf, 0);
            processInternal(tmpcbuf, 0, len, pos);
            return toString();
        }
    }

    public String decode(final char[] cbuf) throws CharacterCodingException {
        return decode(cbuf, 0, cbuf.length);
    }

    public String decode(final char[] cbuf, final int off, final int len)
        throws CharacterCodingException
    {
        int pos = decodingPos(cbuf, off, len);
        if (pos == -1) {
            return new String(cbuf, off, len);
        } else {
            processInternal(cbuf, off, len, pos);
            return toString();
        }
    }

    @Override
    public void process(final char[] buf, final int off, final int len)
        throws CharacterCodingException
    {
        int pos = decodingPos(buf, off, len);
        if (pos == -1) {
            ensureBufCapacity(len);
            System.arraycopy(buf, off, super.buf, 0, len);
            super.len = len;
        } else {
            processInternal(buf, off, len, pos);
        }
    }

    // CSOFF: FinalParameters
    // CSOFF: ParameterNumber
    private void processInternal(
        final char[] cbuf,
        int off,
        final int len,
        final int from)
        throws CharacterCodingException
    {
        ensureBufCapacity(len);
        super.len = 0;
        int end = len + off;
        while (off < from) {
            super.buf[super.len++] = cbuf[off++];
        }
        while (off < end) {
            char c = cbuf[off];
            if (c == '%') {
                int requiredLength = (end - off) / PCT_ELEMENT_LENGTH;
                if (buf.length < requiredLength) {
                    buf = new byte[Math.max(requiredLength, buf.length << 1)];
                }
                int buflen = 0;
                do {
                    if (off + PCT_ELEMENT_LENGTH > end) {
                        throw new MalformedPctException(
                            "Pct sequence is too short");
                    }
                    int high = fromHex(cbuf[++off]);
                    int low = fromHex(cbuf[++off]);
                    ++off;
                    buf[buflen++] = (byte) ((high << HIGH_OFFSET) + low);
                } while (off < end && cbuf[off] == '%');
                decoder.process(buf, 0, buflen);
                decoder.processWith(appender);
            } else {
                if (c == '+' && decodePlus) {
                    super.buf[super.len] = ' ';
                } else {
                    super.buf[super.len] = c;
                }
                ++off;
                ++super.len;
            }
        }
    }
    // CSON: ParameterNumber
    // CSON: FinalParameters

    private void append(final char[] buf, final int off, final int len) {
        System.arraycopy(buf, off, super.buf, super.len, len);
        super.len += len;
    }

    private class Appender
        implements CharArrayVoidProcessor<RuntimeException>
    {
        @Override
        public void process(final char[] buf, final int off, final int len) {
            append(buf, off, len);
        }
    }
}

