package ru.yandex.parser.uri;

import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;

import ru.yandex.charset.Encoder;
import ru.yandex.function.ByteArrayVoidProcessor;
import ru.yandex.function.CharArrayProcessable;
import ru.yandex.function.CharArrayVoidProcessor;
import ru.yandex.function.Processor;
import ru.yandex.util.string.HexStrings;

public class PctEncoder
    extends CharArrayProcessable
    implements CharArrayVoidProcessor<CharacterCodingException>
{
    private static final int PCT_ELEMENT_LENGTH = 3;
    private static final char[] HEX = HexStrings.UPPER.chars();

    private final BytesProcessor bytesProcessor = new BytesProcessor();
    private final Encoder encoder;
    private final boolean[] encodingTable;
    private final boolean encodeSpace;

    public PctEncoder(final PctEncodingRule encondingRule) {
        this(encondingRule, StandardCharsets.UTF_8);
    }

    public PctEncoder(
        final PctEncodingRule encodingRule,
        final Charset charset)
    {
        this(new Encoder(charset), encodingRule);
    }

    public PctEncoder(
        final PctEncodingRule encodingRule,
        final CharsetEncoder charsetEncoder)
    {
        this(new Encoder(charsetEncoder), encodingRule);
    }

    public PctEncoder(
        final Encoder encoder,
        final PctEncodingRule encodingRule)
    {
        this.encoder = encoder;
        encodingTable = encodingRule.encodingTable();
        encodeSpace = !encodingTable['+'];
    }

    @Override
    public void process(final char[] buf, final int off, final int len)
        throws CharacterCodingException
    {
        encoder.process(buf, off, len);
        encoder.processWith(bytesProcessor);
    }

    @Override
    public void ensureMaxBufCapacity(final int maxCapacity) {
        super.ensureMaxBufCapacity(maxCapacity);
        encoder.ensureMaxBufCapacity(maxCapacity);
    }

    private class BytesProcessor
        implements ByteArrayVoidProcessor<RuntimeException>
    {
        @Override
        public void process(final byte[] buf, final int off, final int len) {
            int maxLength = len * PCT_ELEMENT_LENGTH;
            ensureBufCapacity(maxLength);
            int cbuflen = 0;
            char[] cbuf = PctEncoder.this.buf;
            for (int i = 0; i < len; ++i, ++cbuflen) {
                byte b = buf[i];
                if (b > 0 && encodingTable[b]) {
                    cbuf[cbuflen] = (char) b;
                } else {
                    if (b == ' ' && encodeSpace) {
                        cbuf[cbuflen] = '+';
                    } else {
                        cbuf[cbuflen++] = '%';
                        cbuf[cbuflen++] =
                            HEX[(b >>> HexStrings.SHIFT) & HexStrings.MASK];
                        cbuf[cbuflen] = HEX[b & HexStrings.MASK];
                    }
                }
            }
            PctEncoder.this.len = cbuflen;
        }
    }

    public static class CompactEncoder
        extends CharArrayProcessable
        implements Processor<String, String, CharacterCodingException>
    {
        private final int maxCapacity;
        private final PctEncoder encoder;

        public CompactEncoder(
            final int maxCapacity,
            final PctEncodingRule encodingRule)
        {
            this.maxCapacity = maxCapacity;
            encoder = new PctEncoder(encodingRule);
        }

        private void encode(final String str, final int off, final int len)
            throws CharacterCodingException
        {
            ensureBufCapacity(len);
            str.getChars(off, off + len, buf, 0);
            encoder.process(buf, 0, len);
        }

        private void encode(
            final StringBuilder sb,
            final int off,
            final int len)
            throws CharacterCodingException
        {
            ensureBufCapacity(len);
            sb.getChars(off, off + len, buf, 0);
            encoder.process(buf, 0, len);
        }

        private void postProcess() {
            ensureMaxBufCapacity(maxCapacity);
            encoder.ensureMaxBufCapacity(maxCapacity);
        }

        @Override
        public String process(final String str)
            throws CharacterCodingException
        {
            return process(str, 0, str.length());
        }

        @Override
        public String process(final String str, final int off, final int len)
            throws CharacterCodingException
        {
            encode(str, off, len);
            String result = encoder.toString();
            postProcess();
            return result;
        }

        public void process(final String str, final StringBuilder out)
            throws CharacterCodingException
        {
            process(str, 0, str.length(), out);
        }

        public void process(
            final String str,
            final int off,
            final int len,
            final StringBuilder out)
            throws CharacterCodingException
        {
            encode(str, off, len);
            encoder.toStringBuilder(out);
            postProcess();
        }

        public void process(final StringBuilder in, final StringBuilder out)
            throws CharacterCodingException
        {
            process(in, 0, in.length(), out);
        }

        public void process(
            final StringBuilder in,
            final int off,
            final int len,
            final StringBuilder out)
            throws CharacterCodingException
        {
            encode(in, off, len);
            encoder.toStringBuilder(out);
            postProcess();
        }
    }
}

