package ru.yandex.parser.uri;

import java.nio.charset.CharacterCodingException;

import ru.yandex.function.StringBuilderProcessorAdapter;
import ru.yandex.function.StringVoidProcessor;
import ru.yandex.http.util.BadRequestException;

public class QueryConstructor {
    private static final String FAILED_TO_ENCODE = "Failed to encode value '";

    private final StringVoidProcessor<char[], CharacterCodingException>
        encoder =
            new StringVoidProcessor<>(new PctEncoder(PctEncodingRule.QUERY));
    private final StringBuilder sb;
    private final StringBuilderProcessorAdapter adapter;
    private boolean addAmpersand;

    public QueryConstructor(final String uri) {
        this(uri, true);
    }

    public QueryConstructor(final String uri, final boolean addAmpersand) {
        this(new StringBuilder(uri), addAmpersand);
    }

    public QueryConstructor(final StringBuilder sb) {
        this(sb, true);
    }

    public QueryConstructor(
        final StringBuilder sb,
        final boolean addAmpersand)
    {
        this.sb = sb;
        this.addAmpersand = addAmpersand;
        adapter = new StringBuilderProcessorAdapter(sb);
    }

    @Override
    public String toString() {
        return new String(sb);
    }

    public void copyIfPresent(final CgiParamsBase from, final String name)
        throws BadRequestException
    {
        copyIfPresent(from, name, name);
    }

    public void copyIfPresent(
        final CgiParamsBase from,
        final String nameFrom,
        final String nameTo)
        throws BadRequestException
    {
        String value = from.getString(nameFrom, null);
        if (value != null) {
            append(nameTo, value);
        }
    }

    public void copyAll(final CgiParams from, final String name)
        throws BadRequestException
    {
        for (String value : from.getAll(name)) {
            append(name, value);
        }
    }

    public void copy(final CgiParamsBase from, final String name)
        throws BadRequestException
    {
        append(name, from.getString(name));
    }

    private void addAmpersand() {
        if (addAmpersand) {
            sb.append('&');
        } else {
            addAmpersand = true;
        }
    }

    public void append(final String name, final String value)
        throws BadRequestException
    {
        try {
            encoder.process(value);
        } catch (CharacterCodingException e) {
            throw new BadRequestException(
                FAILED_TO_ENCODE + value + "' for parameter " + name,
                e);
        }
        addAmpersand();
        sb.append(name);
        sb.append('=');
        encoder.processWith(adapter);
    }

    public void append(final String arbitraryText) throws BadRequestException {
        try {
            encoder.process(arbitraryText);
        } catch (CharacterCodingException e) {
            throw new BadRequestException(
                FAILED_TO_ENCODE + arbitraryText + '\'',
                e);
        }
        encoder.processWith(adapter);
    }

    public void append(final String name, final int value) {
        addAmpersand();
        sb.append(name);
        sb.append('=');
        sb.append(value);
    }

    public void append(final String name, final long value) {
        addAmpersand();
        sb.append(name);
        sb.append('=');
        sb.append(value);
    }

    public StringBuilder sb() {
        return sb;
    }
}

