package ru.yandex.parser.uri;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;
import java.util.NoSuchElementException;

import ru.yandex.function.CharArrayProcessor;

public class QueryParser implements Iterable<QueryParameter> {
    public static final PctEncodedString NULL =
        new PctEncodedString("true", null);

    public enum Factory
        implements CharArrayProcessor<QueryParser, RuntimeException>
    {
        INSTANCE;

        @Override
        public QueryParser process(
            final char[] buf,
            final int off,
            final int len)
        {
            return new QueryParser(
                buf,
                off,
                len,
                new PctDecoder(true, StandardCharsets.UTF_8));
        }
    }

    private final char[] query;
    private final int begin;
    private final int end;
    private final PctDecoder decoder;

    public QueryParser(final String query) {
        this(query, StandardCharsets.UTF_8);
    }

    public QueryParser(final String query, final Charset charset) {
        this(query, new PctDecoder(true, charset));
    }

    public QueryParser(final String query, final PctDecoder decoder) {
        this(query.toCharArray(), decoder);
    }

    QueryParser(final char[] query, final PctDecoder decoder) {
        this(query, 0, query.length, decoder);
    }

    // CSOFF: ParameterNumber
    QueryParser(
        final char[] query,
        final int off,
        final int len,
        final PctDecoder decoder)
    {
        this.query = query;
        this.begin = off;
        this.end = off + len;
        this.decoder = decoder;
    }
    // CSON: ParameterNumber

    public PctDecoder decoder() {
        return decoder;
    }

    @Override
    public Iterator<QueryParameter> iterator() {
        return new QueryIterator();
    }

    private class QueryIterator implements Iterator<QueryParameter> {
        // points to the first non-& character
        private int pos = begin - 1;

        QueryIterator() {
            incrementPos();
        }

        private void incrementPos() {
            ++pos;
            while (pos < end && query[pos] == '&') {
                ++pos;
            }
        }

        @Override
        public boolean hasNext() {
            return pos < end;
        }

        @Override
        public QueryParameter next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            int start = pos++;
            while (pos < end && query[pos] != '&') {
                ++pos;
            }
            int end = pos;
            incrementPos();
            int middle = start;
            while (middle < end && query[middle] != '=') {
                ++middle;
            }
            String name = new String(query, start, middle - start);
            if (middle == end) {
                return new QueryParameter(name, NULL);
            } else {
                ++middle;
                return new QueryParameter(
                    name,
                    new PctEncodedString(
                        query,
                        middle,
                        end - middle,
                        decoder));
            }
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}

