package ru.yandex.parser.uri;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

import ru.yandex.function.StringBuilderProcessorAdapter;

public class UriParser {
    private final char[] uri;
    private final Charset charset;
    private int pathEnd = 0;
    private int queryStart = 0;
    private int queryEnd = 0;
    private int fragmentStart;
    private PctDecoder pathDecoder = null;
    private PctDecoder queryDecoder = null;
    private PctDecoder fragmentDecoder = null;

    public UriParser(final String uri) {
        this(uri, StandardCharsets.UTF_8);
    }

    public UriParser(final String uri, final Charset charset) {
        this(uri.toCharArray(), charset);
    }

    public UriParser(final StringBuilder uri) {
        this(uri, StandardCharsets.UTF_8);
    }

    public UriParser(final StringBuilder uri, final Charset charset) {
        this(toCharArray(uri), charset);
    }

    private UriParser(final char[] uri, final Charset charset) {
        this.uri = uri;
        this.charset = charset;
        fragmentStart = uri.length;
        while (pathEnd < uri.length) {
            if (uri[pathEnd] == '?') {
                queryStart = pathEnd + 1;
                queryEnd = queryStart;
                while (queryEnd < uri.length) {
                    if (uri[queryEnd] == '#') {
                        fragmentStart = queryEnd + 1;
                        break;
                    }
                    ++queryEnd;
                }
                break;
            } else if (uri[pathEnd] == '#') {
                fragmentStart = pathEnd + 1;
                break;
            }
            ++pathEnd;
        }
    }

    private static char[] toCharArray(final StringBuilder sb) {
        int len = sb.length();
        char[] buf = new char[len];
        sb.getChars(0, len, buf, 0);
        return buf;
    }

    public PctDecoder pathDecoder() {
        if (pathDecoder == null) {
            pathDecoder = new PctDecoder(false, charset);
        }
        return pathDecoder;
    }

    public PctDecoder queryDecoder() {
        if (queryDecoder == null) {
            queryDecoder = new PctDecoder(true, charset);
        }
        return queryDecoder;
    }

    public PctDecoder fragmentDecoder() {
        if (fragmentDecoder == null) {
            fragmentDecoder = new PctDecoder(false, charset);
        }
        return fragmentDecoder;
    }

    public PctEncodedString path() {
        return path(pathDecoder());
    }

    public PctEncodedString path(final PctDecoder decoder) {
        return new PctEncodedString(uri, 0, pathEnd, decoder);
    }

    public PctEncodedString query() {
        return query(queryDecoder());
    }

    public PctEncodedString query(final PctDecoder decoder) {
        return new PctEncodedString(
            uri,
            queryStart,
            queryEnd - queryStart,
            decoder);
    }

    public String queryOrNull() {
        if (queryStart < queryEnd) {
            return query().decodeOrRaw();
        } else {
            return null;
        }
    }

    public PctEncodedString fragment() {
        return fragment(fragmentDecoder());
    }

    public PctEncodedString fragment(final PctDecoder decoder) {
        return new PctEncodedString(
            uri,
            fragmentStart,
            uri.length - fragmentStart,
            decoder);
    }

    public String fragmentOrNull() {
        if (fragmentStart < uri.length) {
            return fragment().decodeOrRaw();
        } else {
            return null;
        }
    }

    public String rawPath() {
        return new String(uri, 0, pathEnd);
    }

    public String rawQuery() {
        return new String(uri, queryStart, queryEnd - queryStart);
    }

    public String rawFragment() {
        return new String(uri, fragmentStart, uri.length - fragmentStart);
    }

    public boolean hasPath() {
        return pathEnd != 0;
    }

    public boolean hasQuery() {
        return queryStart != 0;
    }

    public boolean hasFragment() {
        return fragmentStart < uri.length;
    }

    public PathParser pathParser() {
        return pathParser(pathDecoder());
    }

    public PathParser pathParser(final PctDecoder decoder) {
        return new PathParser(uri, 0, pathEnd, decoder);
    }

    public QueryParser queryParser() {
        return queryParser(queryDecoder());
    }

    public QueryParser queryParser(final PctDecoder decoder) {
        return new QueryParser(
            uri,
            queryStart,
            queryEnd - queryStart,
            decoder);
    }

    public String addOrReplaceCgiParameter(final QueryParameter replacement) {
        StringBuilder uri = new StringBuilder();
        StringBuilderProcessorAdapter adapter =
            new StringBuilderProcessorAdapter(uri);
        path(null).processWith(adapter);
        uri.append('?');
        boolean first = true;
        boolean replaced = false;
        String name = replacement.name();
        PctEncodedString replacementValue = replacement.value();
        QueryParser parser = queryParser(null);
        for (QueryParameter parameter: parser) {
            if (first) {
                first = false;
            } else {
                uri.append('&');
            }
            String parameterName = parameter.name();
            PctEncodedString value;
            if (name.equals(parameterName)) {
                replaced = true;
                value = replacementValue;
            } else {
                value = parameter.value();
            }
            uri.append(parameterName);
            if (value != QueryParser.NULL) {
                uri.append('=');
                value.processWith(adapter);
            }
        }
        if (!replaced) {
            if (!first) {
                uri.append('&');
            }
            uri.append(name);
            if (replacementValue != QueryParser.NULL) {
                uri.append('=');
                replacementValue.processWith(adapter);
            }
        }
        if (hasFragment()) {
            uri.append('#');
            fragment(null).processWith(adapter);
        }
        return new String(uri);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(uri.length);
        sb.append(path().decodeOrRaw());
        if (queryStart < queryEnd) {
            sb.append('?');
            sb.append(query().decodeOrRaw());
        }
        if (fragmentStart < uri.length) {
            sb.append('#');
            sb.append(fragment().decodeOrRaw());
        }
        return sb.toString();
    }
}

