package ru.yandex.jniwrapper;

import java.nio.file.Paths;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BiFunction;

import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.Processor;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.parser.config.IniConfig;

public abstract class JniWrapper
    implements BiFunction<
        String,
        String,
        Processor<byte[], String, JniWrapperException>>,
        GenericAutoCloseable<RuntimeException>
{
    static {
        System.loadLibrary("jniwrapper");
    }

    protected final ReadWriteLock lock = new ReentrantReadWriteLock();
    protected final Lock readLock = lock.readLock();
    protected final Lock writeLock = lock.writeLock();
    private final String configString;
    private final long handle;
    private volatile boolean closed = false;

    protected JniWrapper(final ImmutableJniWrapperConfig config)
        throws JniWrapperException
    {
        configString = config.config();
        try {
            handle = load(
                config.libraryName(),
                config.ctorName(),
                config.dtorName(),
                config.mainName(),
                config.main16Name(),
                config.logrotateName(),
                config.reloadName(),
                config.freeName());
        } catch (RuntimeException e) {
            throw new JniWrapperException(
                "Failed to load library " + config.libraryName()
                + '(' + config.ctorName()
                + ',' + config.dtorName()
                + ',' + config.mainName()
                + ',' + config.main16Name()
                + ',' + config.logrotateName()
                + ',' + config.reloadName()
                + ',' + config.freeName()
                + ')',
                e);
        }
    }

    public static JniWrapper create(
        final ImmutableJniWrapperConfig config,
        final ThreadFactoryConfig threadFactoryConfig)
        throws JniWrapperException
    {
        if (config.workers() > 0) {
            return new ThreadPoolJniWrapper(config, threadFactoryConfig);
        } else {
            return new ThreadSafeJniWrapper(config);
        }
    }

    private void checkClosed() {
        if (closed) {
            throw new IllegalStateException(
                "JniWrapper closed: " + Long.toHexString(handle));
        }
    }

    // CSOFF: ParameterNumber
    private static native long load(
        final String libraryName,
        final String ctorName,
        final String dtorName,
        final String mainName,
        final String main16Name,
        final String logrotateName,
        final String reloadName,
        final String freeName);
    // CSON: ParameterNumber

    private static native void unload(final long handle);

    private static native long createInstance(
        final long handle,
        final String configString);

    protected long createInstance() throws JniWrapperException {
        readLock.lock();
        try {
            checkClosed();
            try {
                return createInstance(handle, configString);
            } catch (RuntimeException e) {
                throw new JniWrapperException(e);
            }
        } finally {
            readLock.unlock();
        }
    }

    private static native void destroyInstance(
        final long handle,
        final long instance);

    protected void destroyInstance(final long instance) {
        destroyInstance(handle, instance);
    }

    // CSOFF: ParameterNumber
    protected String process(
        final long instance,
        final String uri,
        final String metainfo,
        final byte[] buf,
        final int off,
        final int len)
    {
        readLock.lock();
        try {
            checkClosed();
            return process(handle, instance, uri, metainfo, buf, off, len);
        } finally {
            readLock.unlock();
        }
    }

    private static native String process(
        final long handle,
        final long instance,
        final String uri,
        final String metainfo,
        final byte[] buf,
        final int off,
        final int len);
    // CSON: ParameterNumber

    public abstract void logrotate();

    private static native void logrotate(
        final long handle,
        final long instance);

    protected void logrotate(final long instance) {
        readLock.lock();
        try {
            checkClosed();
            logrotate(handle, instance);
        } finally {
            readLock.unlock();
        }
    }

    public abstract void reload();

    private static native void reload(
        final long handle,
        final long instance);

    protected void reload(final long instance) {
        readLock.lock();
        try {
            checkClosed();
            reload(handle, instance);
        } finally {
            readLock.unlock();
        }
    }

    protected void onClose() {
        unload(handle);
    }

    @Override
    public void close() {
        writeLock.lock();
        try {
            if (!closed) {
                closed = true;
                onClose();
            }
        } finally {
            writeLock.unlock();
        }
    }

    private static String getOrDefault(
        final String[] args,
        final int idx,
        final String defaultValue)
    {
        if (idx >= args.length) {
            return defaultValue;
        } else {
            return args[idx];
        }
    }

    public static void main(final String... args) throws Exception {
        System.out.print(
            IOStreamUtils.consume(System.in).processWith(
                create(
                    new ImmutableJniWrapperConfig(
                        new JniWrapperConfigBuilder(
                            new IniConfig(Paths.get(args[0])))),
                    new ThreadFactoryConfig("Jni-")
                        .group(Thread.currentThread().getThreadGroup())
                        .daemon(true))
                    .apply(
                        getOrDefault(args, 1, ""),
                        getOrDefault(args, 2, null))));
    }
}

