package ru.yandex.jniwrapper;

import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;

import ru.yandex.collection.BlockingBlockingQueue;
import ru.yandex.concurrent.BasicLoadedThreadContext;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.LoadedThread;
import ru.yandex.concurrent.LoadedThreadProvider;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.function.ByteArrayProcessor;
import ru.yandex.function.GenericSupplier;
import ru.yandex.function.Processor;
import ru.yandex.function.RuntimeExceptionWrappingConsumer;

public class ThreadPoolJniWrapper extends JniWrapper {
    private static final BiConsumer<Instance, Instance> LOGROTATE =
        (x, y) -> x.logrotate();
    private static final BiConsumer<Instance, Instance> RELOAD =
        (x, y) -> x.reload();

    private final Map<Instance, Instance> instances =
        new ConcurrentHashMap<>();
    private final ThreadPoolExecutor threadPool;
    private volatile boolean closed = false;

    public ThreadPoolJniWrapper(
        final ImmutableJniWrapperConfig config,
        final ThreadFactoryConfig threadFactoryConfig)
        throws JniWrapperException
    {
        super(config);
        threadPool = new ThreadPoolExecutor(
            config.workers(),
            config.workers(),
            1L,
            TimeUnit.MINUTES,
            new BlockingBlockingQueue<>(
                new LifoWaitBlockingQueue<>(config.queueSize())),
            new NamedThreadFactory(
                new ThreadFactoryConfig(
                    new LoadedThreadProvider<>(
                        new BasicLoadedThreadContext<>(
                            new InstanceFactory(),
                            RuntimeExceptionWrappingConsumer.INSTANCE)),
                    threadFactoryConfig)));
        // Check that config is valid
        destroyInstance(createInstance());
    }

    private void checkClosed() {
        if (closed) {
            throw new IllegalStateException("Pool is closed");
        }
    }

    @Override
    public void close() {
        threadPool.shutdownNow();
        writeLock.lock();
        try {
            if (closed) {
                return;
            } else {
                closed = true;
            }
        } finally {
            writeLock.unlock();
        }
        // force wait for all instances to complete their tasks
        instances.forEach((x, y) -> x.close());
        super.close();
    }

    @Override
    public void logrotate() {
        instances.forEach(LOGROTATE);
    }

    @Override
    public void reload() {
        instances.forEach(RELOAD);
    }

    @Override
    public Processor<byte[], String, JniWrapperException> apply(
        final String uri,
        final String metainfo)
    {
        return new JniProcessor(uri, metainfo);
    }

    private class JniProcessor
        implements ByteArrayProcessor<String, JniWrapperException>
    {
        private final String uri;
        private final String metainfo;

        JniProcessor(final String uri, final String metainfo) {
            this.uri = uri;
            this.metainfo = metainfo;
        }

        @Override
        public String process(final byte[] buf, final int off, final int len)
            throws JniWrapperException
        {
            readLock.lock();
            try {
                checkClosed();
                try {
                    return threadPool
                        .submit(
                            new JniProcessorTask(uri, metainfo, buf, off, len))
                        .get();
                } catch (ExecutionException e) {
                    Throwable cause = e.getCause();
                    if (cause instanceof JniWrapperException) {
                        throw (JniWrapperException) cause;
                    } else {
                        throw new JniWrapperException(e);
                    }
                } catch (Throwable t) {
                    throw new JniWrapperException(t);
                }
            } finally {
                readLock.unlock();
            }
        }
    }

    private static class JniProcessorTask implements Callable<String> {
        private final String uri;
        private final String metainfo;
        private final byte[] buf;
        private final int off;
        private final int len;

        // CSOFF: ParameterNumber
        JniProcessorTask(
            final String uri,
            final String metainfo,
            final byte[] buf,
            final int off,
            final int len)
        {
            this.uri = uri;
            this.metainfo = metainfo;
            this.buf = buf;
            this.off = off;
            this.len = len;
        }
        // CSON: ParameterNumber

        @Override
        public String call() throws JniWrapperException {
            try {
                Instance instance = LoadedThread.getThreadLocalResource();
                return instance.process(uri, metainfo, buf, off, len);
            } catch (IllegalArgumentException e) {
                throw new JniWrapperUnprocessableInputException(e);
            } catch (Throwable t) {
                throw new JniWrapperException(t);
            }
        }
    }

    private class Instance implements AutoCloseable {
        private final long ptr;
        private boolean closed = false;

        Instance(final long ptr) {
            this.ptr = ptr;
        }

        public synchronized void logrotate() {
            if (!closed) {
                ThreadPoolJniWrapper.this.logrotate(ptr);
            }
        }

        public synchronized void reload() {
            if (!closed) {
                ThreadPoolJniWrapper.this.reload(ptr);
            }
        }

        @Override
        public synchronized void close() {
            if (!closed) {
                closed = true;
                instances.remove(this);
                destroyInstance(ptr);
            }
        }

        // CSOFF: ParameterNumber
        public synchronized String process(
            final String uri,
            final String metainfo,
            final byte[] buf,
            final int off,
            final int len)
        {
            if (closed) {
                throw new IllegalStateException(
                    "Instance " + Long.toHexString(ptr) + " closed");
            } else {
                readLock.lock();
                try {
                    checkClosed();
                    return ThreadPoolJniWrapper.this.process(
                        ptr,
                        uri,
                        metainfo,
                        buf,
                        off,
                        len);
                } finally {
                    readLock.unlock();
                }
            }
        }
        // CSON: ParameterNumber
    }

    private class InstanceFactory
        implements GenericSupplier<Instance, JniWrapperException>
    {
        @Override
        public Instance get() throws JniWrapperException {
            readLock.lock();
            try {
                checkClosed();
                Instance instance = new Instance(createInstance());
                instances.put(instance, instance);
                return instance;
            } finally {
                readLock.unlock();
            }
        }
    }
}

