#include "main.h"

#include <mail/so/libs/jniwrapper_base/jniwrapper_base.h>

#include <util/generic/ptr.h>
#include <util/generic/scope.h>
#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <util/system/compiler.h>
#include <util/system/defaults.h>
#include <util/system/dynlib.h>

#include <jni.h>

#if defined(RTLD_NOW) && defined(RTLD_LOCAL)
#define JNIWRAPPER_DLLOPEN_FLAGS (RTLD_NOW | RTLD_LOCAL)
#else
#define JNIWRAPPER_DLLOPEN_FLAGS DEFAULT_DLLOPEN_FLAGS
#endif

#define WHEREAMI __FILE__ ":" Y_STRINGIZE(__LINE__) ": "

namespace NPrivate {
    template <typename T>
    static T LoadSymbol(JNIEnv* env, TDynamicLibrary& library, jstring name, T defaultValue) {
        if (name) {
            TString utf8Name{NJniWrapper::JStringToUtf(env, name)};
            return (T) library.Sym(utf8Name.c_str());
        } else {
            return defaultValue;
        }
    }

    class TLoadSymbol {
    private:
        JNIEnv* Env;
        TDynamicLibrary& Library;
        const jstring Name;
        const void* DefaultValue;

    public:
        TLoadSymbol(JNIEnv* env, TDynamicLibrary& library, jstring name, void* defaultValue)
            : Env(env)
            , Library(library)
            , Name(name)
            , DefaultValue(defaultValue)
        {
        }

        template <typename T>
        operator T() const {
            return LoadSymbol<T>(Env, Library, Name, (T) DefaultValue);
        }
    };
}

static ::NPrivate::TLoadSymbol LoadSymbol(
    JNIEnv* env,
    TDynamicLibrary& library,
    jstring name,
    void* defaultValue = nullptr)
{
    return ::NPrivate::TLoadSymbol(env, library, name, defaultValue);
}

struct TLibraryWrapper {
    TDynamicLibrary Library;
    const JniWrapperConstructorFunction Ctor;
    const JniWrapperDestructorFunction Dtor;
    const JniWrapperMainFunction Main;
    const JniWrapperMain16Function Main16;
    const JniWrapperLogrotateFunction Logrotate;
    const JniWrapperReloadFunction Reload;
    const JniWrapperFreeFunction Free;

    TLibraryWrapper(
        JNIEnv* env,
        jstring libraryName,
        jstring ctorName,
        jstring dtorName,
        jstring mainName,
        jstring main16Name,
        jstring logrotateName,
        jstring reloadName,
        jstring freeName)
        : Library(NJniWrapper::JStringToUtf(env, libraryName), JNIWRAPPER_DLLOPEN_FLAGS)
        , Ctor(LoadSymbol(env, Library, ctorName))
        , Dtor(LoadSymbol(env, Library, dtorName))
        , Main(LoadSymbol(env, Library, mainName))
        , Main16(LoadSymbol(env, Library, main16Name))
        , Logrotate(LoadSymbol(env, Library, logrotateName))
        , Reload(LoadSymbol(env, Library, reloadName))
        , Free(LoadSymbol(env, Library, freeName, (void*) free))
    {
    }
};

extern "C" JNIEXPORT jlong JNICALL
Java_ru_yandex_jniwrapper_JniWrapper_load(
    JNIEnv* env,
    jclass,
    jstring libraryName,
    jstring ctorName,
    jstring dtorName,
    jstring mainName,
    jstring main16Name,
    jstring logrotateName,
    jstring reloadName,
    jstring freeName)
{
    try {
        return (jlong) new TLibraryWrapper(
            env,
            libraryName,
            ctorName,
            dtorName,
            mainName,
            main16Name,
            logrotateName,
            reloadName,
            freeName);
    } catch (...) {
        NJniWrapper::RethrowAsJavaException(env);
        return 0;
    }
}

extern "C" JNIEXPORT void JNICALL
Java_ru_yandex_jniwrapper_JniWrapper_unload(
    JNIEnv*,
    jclass,
    jlong libraryPtr)
{
    delete (TLibraryWrapper*) libraryPtr;
}

static const char* Denullify(void* str) {
    if (str) {
        return (const char*) str;
    } else {
        return "";
    }
}

extern "C" JNIEXPORT jlong JNICALL
Java_ru_yandex_jniwrapper_JniWrapper_createInstance(
    JNIEnv* env,
    jclass,
    jlong libraryPtr,
    jstring configString)
{
    TLibraryWrapper* library = (TLibraryWrapper*) libraryPtr;
    if (!library->Ctor) {
        return 0;
    }
    int ret = 0;
    void *out;
    try {
        if (configString) {
            TString config{NJniWrapper::JStringToUtf(env, configString)};
            ret = library->Ctor(config.c_str(), &out);
        } else {
            ret = library->Ctor(nullptr, &out);
        }
        if (ret == 0) {
            return (jlong) out;
        } else {
            Y_SCOPE_EXIT(library, out) { library->Free(out); };
            const char* str = Denullify(out);
            TString message{
                TString::Join(
                    WHEREAMI "Failed to create instance: ",
                    (const char*) str)};
            NJniWrapper::Throw(env, "java/lang/IllegalArgumentException", message);
        }
    } catch (...) {
        NJniWrapper::RethrowAsJavaException(env);
    }
    return ret;
}

extern "C" JNIEXPORT void JNICALL
Java_ru_yandex_jniwrapper_JniWrapper_destroyInstance(
    JNIEnv*,
    jclass,
    jlong libraryPtr,
    jlong objectPtr)
{
    TLibraryWrapper* library = (TLibraryWrapper*) libraryPtr;
    if (library->Dtor) {
        library->Dtor((void*) objectPtr);
    }
}

extern "C" JNIEXPORT void JNICALL
Java_ru_yandex_jniwrapper_JniWrapper_logrotate(
    JNIEnv*,
    jclass,
    jlong libraryPtr,
    jlong objectPtr)
{
    TLibraryWrapper* library = (TLibraryWrapper*) libraryPtr;
    if (library->Logrotate) {
        library->Logrotate((void*) objectPtr);
    }
}

extern "C" JNIEXPORT void JNICALL
Java_ru_yandex_jniwrapper_JniWrapper_reload(
    JNIEnv*,
    jclass,
    jlong libraryPtr,
    jlong objectPtr)
{
    TLibraryWrapper* library = (TLibraryWrapper*) libraryPtr;
    if (library->Reload) {
        library->Reload((void*) objectPtr);
    }
}

static jstring ProcessUtf8(
    JNIEnv* env,
    TLibraryWrapper* library,
    void* instance,
    jstring uriString,
    jstring metainfoString,
    jbyte* buf,
    jsize len)
{
    TString uri{NJniWrapper::JStringToUtf(env, uriString)};
    THolder<TString> metainfo;
    if (metainfoString) {
        metainfo.Reset(new TString{NJniWrapper::JStringToUtf(env, metainfoString)});
    }
    char* out = nullptr;
    int ret = library->Main(
        instance,
        uri.c_str(),
        metainfo ? metainfo->c_str() : nullptr,
        buf,
        len,
        &out);
    Y_SCOPE_EXIT(library, out) { library->Free(out); };
    jstring result = 0;
    if (ret == 0) {
        const char* str = Denullify(out);
        result = NJniWrapper::UtfToJString(env, str);
        if (!result) {
            NJniWrapper::Throw(
                env,
                "java/lang/OutOfMemoryError",
                TString::Join(WHEREAMI "Failed to allocate memory for result string"));
        }
    } else if (ret == -2) {
        NJniWrapper::Throw(env, "java/lang/IllegalArgumentException", out);
    } else {
        NJniWrapper::Throw(env, "java/lang/RuntimeException", out);
    }
    return result;
}

static const wchar16* Denullify(const wchar16* str, size_t* outLen) {
    if (str) {
        return str;
    } else {
        *outLen = 0;
        return u"";
    }
}

static jstring ProcessUtf16(
    JNIEnv* env,
    TLibraryWrapper* library,
    void* instance,
    jstring uriString,
    jstring metainfoString,
    jbyte* buf,
    jsize len)
{
    TUtf16String uri{NJniWrapper::ConvertJString(env, uriString)};
    THolder<TUtf16String> metainfo;
    const unsigned short* metainfoPtr = nullptr;
    size_t metainfoLen = 0;
    if (metainfoString) {
        metainfo.Reset(new TUtf16String{NJniWrapper::ConvertJString(env, metainfoString)});
        metainfoPtr = (const unsigned short*) metainfo->Data();
        metainfoLen = metainfo->Size();
    }
    wchar16* out = 0;
    size_t outLen = 0;
    int ret = library->Main16(
        instance,
        (const unsigned short*) uri.Data(),
        uri.Size(),
        metainfoPtr,
        metainfoLen,
        buf,
        len,
        (unsigned short**) (void*) &out,
        &outLen);
    Y_SCOPE_EXIT(library, out) { library->Free(out); };
    const wchar16* str = Denullify(out, &outLen);
    jstring result = 0;
    if (ret == 0) {
        result = env->NewString((const jchar*) str, outLen);
        if (!result) {
            NJniWrapper::Throw(
                env,
                "java/lang/OutOfMemoryError",
                TString::Join(WHEREAMI "Failed to allocated memory for result string"));
        }
    } else if (ret == -2) {
        NJniWrapper::Throw(env, "java/lang/IllegalArgumentException", out, outLen);
    } else {
        NJniWrapper::Throw(env, "java/lang/RuntimeException", out, outLen);
    }
    return result;
}

extern "C" JNIEXPORT jstring JNICALL
Java_ru_yandex_jniwrapper_JniWrapper_process(
    JNIEnv* env,
    jclass,
    jlong libraryPtr,
    jlong objectPtr,
    jstring uriString,
    jstring metainfoString,
    jbyteArray input,
    jsize off,
    jsize len)
{
    THolder<jbyte, TFree> buf;
    if (len) {
        buf.Reset((jbyte*) malloc(len));
        if (!buf) {
            NJniWrapper::Throw(
                env,
                "java/lang/OutOfMemoryError",
                "Can't allocate memory for buffer");
            return 0;
        }
        env->GetByteArrayRegion(input, off, len, buf.Get());
        if (env->ExceptionOccurred()) {
            return 0;
        }
    }
    TLibraryWrapper* library = (TLibraryWrapper*) libraryPtr;
    void* instance = (void*) objectPtr;
    jstring result = nullptr;
    try {
        if (library->Main) {
            result = ProcessUtf8(
                env,
                library,
                instance,
                uriString,
                metainfoString,
                buf.Get(),
                len);
        } else {
            result = ProcessUtf16(
                env,
                library,
                instance,
                uriString,
                metainfoString,
                buf.Get(),
                len);
        }
    } catch (...) {
        NJniWrapper::RethrowAsJavaException(env);
    }
    return result;
}

