#pragma once

#include <stddef.h> // for size_t

/**
 * JniWrapper — easy way to adapt C libraries with JNI-calls
 * Provides simple interface to create processor and pass data to its
 * processing function
 * JniWrapper automates processor construction, life time management, data
 * conversion from Java types to native C types
 * Common workflow is:
 *  1. Create processor using constructor function, save pointer
 *  2. Accept data in main function along with processor pointer and return
 *     result as string
 *  3. Repeat 2. until stop signal received
 *  4. Destroy processor using destructor function
 *
 * During program life time processor object could receive logrotate and
 * reload commands for log files reopening and config reloading accordingly.
 *
 * Below you can find description of all functions that could be loaded from
 * your library and referenced by loader
 *
 * Thread safety:
 *  1. Dedending on JniWrapper config, single processor can be created only
 *     once and used by all threads for data processing. Alternatively, one
 *     processor instance will be created for each execution thread (on demand)
 *     plus one instance will be created and destroyed on startup in order to
 *     validate config
 *  2. logrorate and reload functions will be called for all created instances
 *     when requested
 *  3. destructor execution guarded by write lock, while all other operations
 *     guarded by corresponding read lock, so no destruction will occur in the
 *     middle of data processing or logrotate
 */

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Processor construction function
 * Optional. When missing, processing function will received NULL for object
 * Accepts config string or config file path, constructs the processing object
 * and returns it
 * Warning: error description should contain only characters from
 * [Modified UTF-8](https://en.wikipedia.org/wiki/UTF-8#Modified_UTF-8)
 *
 * config       (in)        config string or config file path, may be null if
 *                          not set in JniWrapper config
 *
 * out          (out)       pointer to object constructed or string pointer
 *                          with error description if error occured
 *
 *              (returns)   0 if everything is OK and *out contains pointer to
 *                          processor object
 *                          -1 if error occured, *out will contain pointer to
 *                          null-terminated string of chars containing error
 *                          message. *out will be NULL on out of memory
 *                          -2 if argument is invalid, *out will contain
 *                          pointer to null-terminated string of chars
 *                          containing error message
 */
typedef int (*JniWrapperConstructorFunction)(const char* config, void** out);

/**
 * Processor destructor function
 * Optional. Gently destroys processor object and release resources
 *
 * object       (in)        pointer to processor object, may be NULL if
 *                          constructor is missing
 */
typedef void (*JniWrapperDestructorFunction)(void* object);

/**
 * Main data processing function, accepts data, some additional info (optional)
 * and returns null-terminating string of chars containing processing result
 *
 * object       (in)        pointer to processor object, may be NULL if
 *                          constructor is missing
 *
 * uri          (in)        when JniWrapper executes as part of HTTP wrapper
 *                          (a.k.a. cokemulator), uri contains request uri
 *
 * metainfo     (in)        when cokemulator works in e-mail text part
 *                          extracting mode, metainfo will contain mimetype and
 *                          encoding, for example: text/html; charset="utf-8",
 *                          otherwise, contains Content-Type: header value
 *
 * data         (in)        pointer to data to process, may be NULL
 *
 * size         (in)        data block length
 *
 * out          (out)       pointer to null-terminated string of chars with
 *                          processing result or error description, depending
 *                          on return value
 *                          this string will be deallocated by caller
 *
 *              (returns)   0 if everything is OK and *out contains pointer to
 *                          result string
 *                          -1 if error occured, *out will contain pointer to
 *                          null-terminated string of chars containing error
 *                          message. *out will be NULL on out of memory
 *                          -2 if argument is invalid, *out will contain
 *                          pointer to null-terminated string of chars
 *                          containing error message
 *                          Warning: error description should contain only
 *                          characters from
 *                          [Modified UTF-8](https://en.wikipedia.org/wiki/UTF-8#Modified_UTF-8)
 */
typedef int (*JniWrapperMainFunction)(
    void* object,
    const char* uri,
    const char* metainfo,
    const void* data,
    size_t size,
    char** out);

/**
 * Main data processing function, accepts data, some additional info (optional)
 * and returns null-terminating string of UTF-16 chars containing processing
 * result
 *
 * object       (in)        pointer to processor object, may be NULL if
 *                          constructor is missing
 *
 * uri          (in)        when JniWrapper executes as part of HTTP wrapper
 *                          (a.k.a. cokemulator), uri contains request uri in
 *                          UTF-16 charset
 *
 * uriLen       (in)        uri length in UTF-16 characters
 *
 * metainfo     (in)        when cokemulator works in e-mail text part
 *                          extracting mode, metainfo will contain mimetype and
 *                          encoding in UTF-16 charset,
 *                          for example: text/html; charset="utf-8",
 *                          otherwise, contains Content-Type: header value
 *
 * metainfoLen  (in)        metainfo length in UTF-16 characters
 *
 * data         (in)        pointer to data to process, may be NULL
 *
 * size         (in)        data block length
 *
 * out          (out)       pointer to null-terminated string of UTF-16 chars
 *                          with processing result or error description,
 *                          depending on return value.
 *                          this string will be deallocated by caller
 *
 * outLen       (out)       output string length in UTF-16 characters
 *
 *              (returns)   0 if everything is OK and *out contains pointer to
 *                          result string
 *                          -1 if error occured, *out will contain pointer to
 *                          null-terminated string of UTF-16 chars containing
 *                          error message. *out will be NULL on out of memory
 *                          -2 if argument is invalid, *out will contain
 *                          pointer to null-terminated string of UTF-16 chars
 *                          containing error message
 *                          Warning: error description should contain only
 *                          characters from
 *                          [Modified UTF-8](https://en.wikipedia.org/wiki/UTF-8#Modified_UTF-8)
 */
typedef int (*JniWrapperMain16Function)(
    void* object,
    const unsigned short* uri,
    const size_t uriLen,
    const unsigned short* metainfo,
    const size_t metainfoLen,
    const void* data,
    size_t size,
    unsigned short** out,
    size_t* outLen);

/**
 * Logrotate function
 * Optional. Called when logs reopen was requested by external source (like
 * logrotate script).
 *
 * object       (in)        pointer to processor object, may be NULL if
 *                          constructor is missing
 */
typedef void (*JniWrapperLogrotateFunction)(void* object);

/**
 * Config reload function
 * Optional. Called when config reloading was requested by external source
 * (like notify script on dynamic resource update)
 *
 * object       (in)        pointer to processor object, may be NULL if
 *                          constructor is missing
 */
typedef void (*JniWrapperReloadFunction)(void* object);

/**
 * Memory deallocation function
 * Optional. Release memory previously allocated by constructor (for error
 * description) or main function
 * If functions is missing, libc `free' function will be used
 *
 * object       (in)        pointer to previously allocated memory area
 */
typedef void (*JniWrapperFreeFunction)(void* object);

#ifdef __cplusplus
}
#endif

