package ru.yandex.json.async.consumer;

import java.io.IOException;
import java.nio.CharBuffer;
import java.util.Collection;
import java.util.function.LongFunction;
import java.util.function.Supplier;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.nio.IOControl;
import org.apache.http.protocol.HttpContext;

import ru.yandex.function.GenericConsumer;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.AsyncCharConsumer;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.parser.StringCollectors;

public class JsonStreamAsyncConsumer<T, C extends Collection<T>>
    extends AsyncCharConsumer<C>
    implements GenericConsumer<JsonObject, JsonException>
{
    private final GenericFunction<JsonObject, T, JsonException> consumer;
    private final JsonParser parser;
    private final C results;
    private IOException exception = null;

    // CSOFF: ParameterNumber
    public JsonStreamAsyncConsumer(
        final HttpEntity entity,
        final GenericFunction<JsonObject, T, JsonException> consumer,
        final Supplier<C> storageFactory,
        final LongFunction<StringCollectors> stringCollectorsFactory,
        final ContainerFactory containerFactory)
        throws HttpException
    {
        super(entity);
        this.results = storageFactory.get();
        this.consumer = consumer;
        parser = new JsonParser(
            new StackContentHandler(
                new TypesafeValueContentHandler(
                    this,
                    stringCollectorsFactory.apply(contentLength(entity)),
                    containerFactory)),
            true);
    }
    // CSON: ParameterNumber

    @Override
    protected void consumeContent(
        final CharBuffer buf,
        final IOControl ioctrl)
        throws IOException
    {
        try {
            parser.process(buf.array(), buf.arrayOffset(), buf.remaining());
        } catch (Throwable t) {
            // we need to read the content completely
            exception = new IOException(t);
            exception.printStackTrace();
        }
    }

    @Override
    public void accept(final JsonObject json) throws JsonException {
        if (exception == null) {
            T result = consumer.apply(json);
            if (result != null) {
                results.add(result);
            }
        }
    }

    @Override
    protected C buildResult(final HttpContext context) throws Exception {
        if (exception != null) {
            throw new BadRequestException(exception);
        }

        parser.eof();
        return results;
    }
}
