package ru.yandex.json.dom;

import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;

import ru.yandex.charset.StreamEncoder;
import ru.yandex.json.parser.ContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;

public class CopyHandler implements ContentHandler {
    private final StringBuilder sb = new StringBuilder();
    private final JsonWriter writer;
    private boolean inString = false;

    public CopyHandler(final JsonWriter writer) {
        this.writer = writer;
    }

    @Override
    public void startObject() throws JsonException {
        try {
            writer.startObject();
        } catch (IOException e) {
            throw new JsonException(e);
        }
    }

    @Override
    public void endObject() throws JsonException {
        try {
            writer.endObject();
        } catch (IOException e) {
            throw new JsonException(e);
        }
    }

    @Override
    public void startArray() throws JsonException {
        try {
            writer.startArray();
        } catch (IOException e) {
            throw new JsonException(e);
        }
    }

    @Override
    public void endArray() throws JsonException {
        try {
            writer.endArray();
        } catch (IOException e) {
            throw new JsonException(e);
        }
    }

    // CSOFF: ParameterNumber
    @Override
    public void key(
        final char[] buf,
        final int off,
        final int len,
        final boolean eol)
        throws JsonException
    {
        try {
            sb.append(buf, off, len);
            if (eol) {
                writer.key(sb.toString());
                sb.setLength(0);
            }
        } catch (IOException e) {
            throw new JsonException(e);
        }
    }

    @Override
    public void value(
        final char[] buf,
        final int off,
        final int len,
        final boolean eol)
        throws JsonException
    {
        try {
            if (!inString) {
                inString = true;
                writer.startString();
            }
            writer.write(buf, off, len);
            if (eol) {
                writer.endString();
                inString = false;
            }
        } catch (IOException e) {
            throw new JsonException(e);
        }
    }
    // CSON: ParameterNumber

    @Override
    public void value(final long value) throws JsonException {
        try {
            writer.value(value);
        } catch (IOException e) {
            throw new JsonException(e);
        }
    }

    @Override
    public void value(final double value) throws JsonException {
        try {
            writer.value(value);
        } catch (IOException e) {
            throw new JsonException(e);
        }
    }

    @Override
    public void value(final boolean value) throws JsonException {
        try {
            writer.value(value);
        } catch (IOException e) {
            throw new JsonException(e);
        }
    }

    @Override
    public void nullValue() throws JsonException {
        try {
            writer.nullValue();
        } catch (IOException e) {
            throw new JsonException(e);
        }
    }

    @Override
    public void endObjectEntry() {
    }

    public static void main(final String... args) throws Exception {
        try (JsonWriter writer =
                JsonTypeExtractor.NORMAL
                    .extract(args[0], false)
                    .create(
                        new StreamEncoder(
                            System.out,
                            StandardCharsets.UTF_8)))
        {
            new JsonParser(new CopyHandler(writer))
                .parse(
                    new InputStreamReader(
                        System.in,
                        StandardCharsets.UTF_8));
        }
    }
}

