package ru.yandex.json.dom;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import ru.yandex.collection.FilterList;
import ru.yandex.function.GenericConsumer;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriterBase;

public class JsonList
    extends FilterList<JsonObject, List<JsonObject>>
    implements GenericConsumer<JsonObject, JsonException>, JsonObject
{
    public static final JsonList EMPTY = new JsonList(
        BasicContainerFactory.INSTANCE,
        Collections.emptyList());

    private final ContainerFactory containerFactory;

    public JsonList(final ContainerFactory containerFactory) {
        this(containerFactory, containerFactory.createArrayContainer());
    }

    public JsonList(
        final ContainerFactory containerFactory,
        final int initialCapacity)
    {
        this(
            containerFactory,
            containerFactory.createArrayContainer(initialCapacity));
    }

    JsonList(
        final ContainerFactory containerFactory,
        final List<JsonObject> list)
    {
        super(list);
        this.containerFactory = containerFactory;
    }

    public ContainerFactory containerFactory() {
        return containerFactory;
    }

    @Override
    public void accept(final JsonObject value) {
        add(value);
    }

    @Override
    public Type type() {
        return Type.LIST;
    }

    public JsonList deepCopy() {
        return deepCopy(containerFactory);
    }

    @Override
    public JsonList deepCopy(final ContainerFactory containerFactory) {
        int size = size();
        List<JsonObject> list = containerFactory.createArrayContainer(size);
        for (int i = 0; i < size; ++i) {
            list.add(get(i).deepCopy(containerFactory));
        }
        return new JsonList(containerFactory, list);
    }

    public JsonList filter(final JsonObjectFilter filter) {
        return filter(filter, containerFactory);
    }

    @Override
    public JsonList filter(
        final JsonObjectFilter filter,
        final ContainerFactory containerFactory)
    {
        int size = size();
        List<JsonObject> list = containerFactory.createArrayContainer(size);
        for (int i = 0; i < size; ++i) {
            JsonObject value = get(i).filter(filter, containerFactory);
            if (filter.test(value)) {
                list.add(value);
            }
        }
        return new JsonList(containerFactory, list);
    }

    @Override
    public JsonList asList() {
        return this;
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startArray();
        for (JsonObject object: this) {
            object.writeValue(writer);
        }
        writer.endArray();
    }

    // null masking List member-functions
    @Override
    public boolean add(final JsonObject object) {
        return super.add(JsonMap.maskNull(object));
    }

    @Override
    public void add(final int index, final JsonObject object) {
        super.add(index, JsonMap.maskNull(object));
    }

    @Override
    public boolean addAll(final Collection<? extends JsonObject> c) {
        for (JsonObject object: c) {
            add(object);
        }
        return !c.isEmpty();
    }

    @Override
    public boolean addAll(
        final int index,
        final Collection<? extends JsonObject> c)
    {
        int pos = index;
        for (JsonObject object: c) {
            add(pos++, object);
        }
        return pos != index;
    }

    @Override
    public JsonObject remove(final int index) {
        return JsonMap.maskNull(super.remove(index));
    }

    @Override
    public JsonObject set(final int index, final JsonObject object) {
        return super.set(index, JsonMap.maskNull(object));
    }

    @Override
    public JsonList subList(final int fromIndex, final int toIndex) {
        return new JsonList(
            containerFactory,
            super.subList(fromIndex, toIndex));
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public boolean equals(final Object o) {
        return super.equals(o);
    }
}

