package ru.yandex.json.dom;

import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;

import ru.yandex.collection.FilterMap;
import ru.yandex.collection.SingletonIterator;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.string.ValuesStorage;

public class JsonMap
    extends FilterMap<String, JsonObject, Map<String, JsonObject>>
    implements JsonObject, ValuesStorage<JsonException>
{
    public static final JsonMap EMPTY = new JsonMap(
        BasicContainerFactory.INSTANCE,
        Collections.emptyMap());

    private final ContainerFactory containerFactory;

    public JsonMap(final ContainerFactory containerFactory) {
        this(containerFactory, containerFactory.createObjectContainer());
    }

    public JsonMap(
        final ContainerFactory containerFactory,
        final int capacity)
    {
        this(
            containerFactory,
            containerFactory.createObjectContainer(capacity));
    }

    JsonMap(
        final ContainerFactory containerFactory,
        final Map<String, JsonObject> map)
    {
        super(map);
        this.containerFactory = containerFactory;
    }

    public static JsonObject maskNull(final JsonObject object) {
        if (object == null) {
            return JsonNull.INSTANCE;
        } else {
            return object;
        }
    }

    public static JsonMap singletonMap(
        final String name,
        final JsonObject value)
    {
        return new JsonMap(
            BasicContainerFactory.INSTANCE,
            Collections.singletonMap(name, value));
    }

    public ContainerFactory containerFactory() {
        return containerFactory;
    }

    @Override
    public Type type() {
        return Type.MAP;
    }

    public JsonMap deepCopy() {
        return deepCopy(containerFactory);
    }

    @Override
    public JsonMap deepCopy(final ContainerFactory containerFactory) {
        Map<String, JsonObject> map =
            containerFactory.createObjectContainer(size());
        for (Map.Entry<String, JsonObject> entry: entrySet()) {
            map.put(
                entry.getKey(),
                entry.getValue().deepCopy(containerFactory));
        }
        return new JsonMap(containerFactory, map);
    }

    public JsonMap filter(final JsonObjectFilter filter) {
        return filter(filter, containerFactory);
    }

    @Override
    public JsonMap filter(
        final JsonObjectFilter filter,
        final ContainerFactory containerFactory)
    {
        Map<String, JsonObject> map =
            containerFactory.createObjectContainer(size());
        for (Map.Entry<String, JsonObject> entry: entrySet()) {
            JsonObject value =
                entry.getValue().filter(filter, containerFactory);
            String key = entry.getKey();
            if (filter.test(key, value)) {
                map.put(key, value);
            }
        }
        return new JsonMap(containerFactory, map);
    }

    @Override
    public JsonMap asMap() {
        return this;
    }

    @Override
    public JsonObject get(final Object key) {
        return maskNull(super.get(key));
    }

    @Override
    public JsonObject remove(final Object key) {
        return maskNull(super.remove(key));
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        for (Map.Entry<String, JsonObject> entry: entrySet()) {
            writer.key(entry.getKey());
            entry.getValue().writeValue(writer);
        }
        writer.endObject();
    }

    // null masking Map member-functions
    @Override
    public JsonObject put(final String key, final JsonObject value) {
        return maskNull(super.put(key, maskNull(value)));
    }

    @Override
    public void putAll(final Map<? extends String, ? extends JsonObject> m) {
        for (Map.Entry<? extends String, ? extends JsonObject> entry
                : m.entrySet())
        {
            super.put(entry.getKey(), maskNull(entry.getValue()));
        }
    }

    @Override
    public JsonException parameterNotSetException(final String name) {
        return new JsonException("Attribute " + name + " is not set");
    }

    @Override
    public JsonException parseFailedException(
        final String name,
        final String value,
        final Throwable cause)
    {
        return new JsonException(
            "Failed to parse parameter " + name
            + " with value '" + value + '\'',
            cause);
    }

    @Override
    public String getOrNull(final String name) {
        JsonObject object = super.get(name);
        if (object != null) {
            switch (object.type()) {
                case NULL:
                case LIST:
                case MAP:
                    break;

                default:
                    try {
                        return object.asString();
                    } catch (JsonBadCastException e) {
                        // impossible case
                        throw new RuntimeException(e);
                    }
            }
        }
        return null;
    }

    @Override
    public String getLastOrNull(final String name) {
        // Exactly one value per key
        return getOrNull(name);
    }

    @Override
    public Iterator<String> getAllOrNull(final String name) {
        String value = getOrNull(name);
        if (value == null) {
            return null;
        } else {
            return new SingletonIterator<>(value);
        }
    }

    // Speed optimizations in order to avoid conversions long -> String -> long
    @Override
    public boolean getBoolean(final String name) throws JsonException {
        JsonObject object = super.get(name);
        if (object != null) {
            switch (object.type()) {
                case BOOLEAN:
                case LONG:
                case STRING:
                    try {
                        return object.asBoolean();
                    } catch (Exception e) {
                        throw parseFailedException(name, object.toString(), e);
                    }
                default:
                    break;
            }
        }
        return ValuesStorage.super.getBoolean(name);
    }

    @Override
    public long getLong(final String name) throws JsonException {
        JsonObject object = super.get(name);
        if (object != null) {
            switch (object.type()) {
                case DOUBLE:
                case LONG:
                case STRING:
                    try {
                        return object.asLong();
                    } catch (Exception e) {
                        throw parseFailedException(name, object.toString(), e);
                    }
                default:
                    break;
            }
        }
        return ValuesStorage.super.getLong(name);
    }

    @Override
    public Long getLong(final String name, final Long defaultValue)
        throws JsonException
    {
        JsonObject object = super.get(name);
        if (object == null) {
            return defaultValue;
        } else {
            switch (object.type()) {
                case DOUBLE:
                case LONG:
                case STRING:
                    try {
                        return object.asLong();
                    } catch (Exception e) {
                        throw parseFailedException(name, object.toString(), e);
                    }
                default:
                    break;
            }
        }
        return ValuesStorage.super.getLong(name, defaultValue);
    }

    @Override
    public double getDouble(final String name) throws JsonException {
        JsonObject object = super.get(name);
        if (object != null) {
            switch (object.type()) {
                case DOUBLE:
                case LONG:
                case STRING:
                    try {
                        return object.asDouble();
                    } catch (Exception e) {
                        throw parseFailedException(name, object.toString(), e);
                    }
                default:
                    break;
            }
        }
        return ValuesStorage.super.getDouble(name);
    }

    public JsonList getList(final String name) throws JsonException {
        JsonObject object = super.get(name);
        if (object == null || object == JsonNull.INSTANCE) {
            throw parameterNotSetException(name);
        } else {
            try {
                return object.asList();
            } catch (Exception e) {
                throw parseFailedException(name, object.toString(), e);
            }
        }
    }

    public JsonMap getMap(final String name) throws JsonException {
        JsonObject object = super.get(name);
        if (object == null || object == JsonNull.INSTANCE) {
            throw parameterNotSetException(name);
        } else {
            try {
                return object.asMap();
            } catch (Exception e) {
                throw parseFailedException(name, object.toString(), e);
            }
        }
    }

    public JsonList getListOrNull(final String name) throws JsonException {
        JsonObject object = super.get(name);
        if (object == null || object == JsonNull.INSTANCE) {
            return null;
        } else {
            try {
                return object.asList();
            } catch (Exception e) {
                throw parseFailedException(name, object.toString(), e);
            }
        }
    }

    public JsonMap getMapOrNull(final String name) throws JsonException {
        JsonObject object = super.get(name);
        if (object == null || object == JsonNull.INSTANCE) {
            return null;
        } else {
            try {
                return object.asMap();
            } catch (Exception e) {
                throw parseFailedException(name, object.toString(), e);
            }
        }
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public boolean equals(final Object o) {
        return super.equals(o);
    }
}

