package ru.yandex.json.dom;

import java.util.List;
import java.util.Map;

import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;

public interface JsonObject extends JsonValue {
    enum Type {
        NULL,
        BOOLEAN,
        LONG,
        DOUBLE,
        STRING,
        LIST,
        MAP
    }

    Type type();

    boolean isEmpty();

    JsonObject deepCopy(ContainerFactory containerFactory);

    // Creates deep copy of current object but some map and list entries may be
    // removed by filter
    JsonObject filter(
        JsonObjectFilter filter,
        ContainerFactory containerFactory);

    default boolean asBoolean() throws JsonBadCastException {
        throw new JsonBadCastException(this, "boolean");
    }

    default Boolean asBooleanOrNull() throws JsonBadCastException {
        return asBoolean();
    }

    default long asLong() throws JsonBadCastException {
        throw new JsonBadCastException(this, "long");
    }

    default Long asLongOrNull() throws JsonBadCastException {
        return asLong();
    }

    default double asDouble() throws JsonBadCastException {
        throw new JsonBadCastException(this, "double");
    }

    default Double asDoubleOrNull() throws JsonBadCastException {
        return asDouble();
    }

    default String asString() throws JsonBadCastException {
        throw new JsonBadCastException(this, "String");
    }

    default String asStringOrNull() throws JsonBadCastException {
        return asString();
    }

    default JsonList asList() throws JsonBadCastException {
        throw new JsonBadCastException(this, "List");
    }

    default JsonList asListOrNull() throws JsonException {
        return asList();
    }

    default JsonMap asMap() throws JsonBadCastException {
        throw new JsonBadCastException(this, "Map");
    }

    default JsonMap asMapOrNull() throws JsonBadCastException {
        return asMap();
    }

    default JsonObject get(int index) throws JsonBadCastException {
        throw new JsonBadCastException(this, "List with index " + index);
    }

    default JsonObject get(Object key) throws JsonBadCastException {
        throw new JsonBadCastException(this, "Map with key <" + key + '>');
    }

    static JsonObject adapt(
        final ContainerFactory containerFactory,
        final Object object)
        throws JsonException
    {
        JsonObject result;
        if (object == null) {
            result = JsonNull.INSTANCE;
        } else if (object instanceof String) {
            result = new JsonString((String) object);
        } else if (object instanceof Number) {
            Number number = (Number) object;
            if (number instanceof Double || number instanceof Float) {
                result = new JsonDouble(number.doubleValue());
            } else {
                result = new JsonLong(number.longValue());
            }
        } else if (object instanceof Boolean) {
            if (((Boolean) object).booleanValue()) {
                result = JsonBoolean.TRUE;
            } else {
                result = JsonBoolean.FALSE;
            }
        } else if (object instanceof Map) {
            Map<?, ?> map = (Map<?, ?>) object;
            Map<String, JsonObject> resultMap =
                containerFactory.createObjectContainer(map.size());
            for (Map.Entry<?, ?> entry: map.entrySet()) {
                Object key = entry.getKey();
                if (key instanceof String) {
                    resultMap.put(
                        (String) key,
                        adapt(containerFactory, entry.getValue()));
                } else {
                    Class<?> clazz;
                    if (key == null) {
                        clazz = null;
                    } else {
                        clazz = key.getClass();
                    }
                    throw new JsonException(
                        "Unable to adapt map key " + key
                        + " of class " + clazz);
                }
            }
            result = new JsonMap(containerFactory, resultMap);
        } else if (object instanceof List) {
            List<?> list = (List<?>) object;
            List<JsonObject> resultList =
                containerFactory.createArrayContainer(list.size());
            for (Object obj: list) {
                resultList.add(adapt(containerFactory, obj));
            }
            result = new JsonList(containerFactory, resultList);
        } else {
            throw new JsonException(
                "Unable to adapt value " + object
                + " of " + "class " + object.getClass());
        }
        return result;
    }
}

