package ru.yandex.json.dom;

import java.io.IOException;
import java.io.Reader;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.function.GenericConsumer;
import ru.yandex.function.Processable;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.parser.StackedContentHandler;
import ru.yandex.json.parser.StringCollectors;
import ru.yandex.json.parser.StringCollectorsFactory;

public class TypesafeValueContentHandler extends StackedContentHandler {
    protected final GenericConsumer<JsonObject, ? extends JsonException>
        consumer;
    protected final StringCollectors stringCollectors;
    protected final ContainerFactory containerFactory;

    public TypesafeValueContentHandler(
        final GenericConsumer<JsonObject, ? extends JsonException> consumer)
    {
        this(consumer, StringCollectorsFactory.INSTANCE.apply(-1L));
    }

    public TypesafeValueContentHandler(
        final GenericConsumer<JsonObject, ? extends JsonException> consumer,
        final StringCollectors stringCollectors)
    {
        this(
            consumer,
            stringCollectors,
            BasicContainerFactory.INSTANCE);
    }

    public TypesafeValueContentHandler(
        final GenericConsumer<JsonObject, ? extends JsonException> consumer,
        final StringCollectors stringCollectors,
        final ContainerFactory containerFactory)
    {
        this.consumer = consumer;
        this.stringCollectors = stringCollectors;
        this.containerFactory = containerFactory;
    }

    public static JsonParser prepareParser(
        final GenericConsumer<JsonObject, ? extends JsonException> consumer)
    {
        return new JsonParser(
            new StackContentHandler(
                new TypesafeValueContentHandler(consumer)));
    }

    public static JsonObject parse(final String str) throws JsonException {
        BasicGenericConsumer<JsonObject, JsonException> consumer =
            new BasicGenericConsumer<>();
        prepareParser(consumer).parse(str);
        return consumer.get();
    }

    public static JsonObject parse(final Reader reader)
        throws IOException, JsonException
    {
        BasicGenericConsumer<JsonObject, JsonException> consumer =
            new BasicGenericConsumer<>();
        prepareParser(consumer).parse(reader);
        return consumer.get();
    }

    public static JsonObject parse(final Processable<char[]> data)
        throws JsonException
    {
        BasicGenericConsumer<JsonObject, JsonException> consumer =
            new BasicGenericConsumer<>();
        JsonParser parser = prepareParser(consumer);
        data.processWith(parser);
        parser.eof();
        return consumer.get();
    }

    @Override
    public void startObject() {
        stackContentHandler.push(
            new TypesafeObjectContentHandler(
                consumer,
                stringCollectors,
                containerFactory));
    }

    @Override
    public void startArray() {
        stackContentHandler.push(
            new TypesafeArrayContentHandler(
                stringCollectors,
                containerFactory,
                consumer));
    }

    // CSOFF: ParameterNumber
    @Override
    public void value(
        final char[] buf,
        final int off,
        final int len,
        final boolean eol)
        throws JsonException
    {
        if (eol) {
            consumer.accept(
                new JsonString(
                    stringCollectors.valueStringCollector()
                        .appendLast(buf, off, len)));
        } else {
            stringCollectors.valueStringCollector().append(buf, off, len);
        }
    }
    // CSON: ParameterNumber

    @Override
    public void value(final long value) throws JsonException {
        consumer.accept(JsonLong.valueOf(value));
    }

    @Override
    public void value(final double value) throws JsonException {
        consumer.accept(new JsonDouble(value));
    }

    @Override
    public void value(final boolean value) throws JsonException {
        consumer.accept(JsonBoolean.valueOf(value));
    }

    @Override
    public void nullValue() throws JsonException {
        consumer.accept(JsonNull.INSTANCE);
    }
}

