package ru.yandex.json.dom;

import java.util.List;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.function.GenericConsumer;
import ru.yandex.function.Processable;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.parser.StackedContentHandler;
import ru.yandex.json.parser.StringCollector;
import ru.yandex.json.parser.StringCollectorFactory;

public class ValueContentHandler extends StackedContentHandler {
    protected final GenericConsumer<Object, ? extends JsonException> consumer;
    protected final StringCollector stringCollector;
    protected final ContainerFactory containerFactory;

    public ValueContentHandler(
        final GenericConsumer<Object, ? extends JsonException> consumer)
    {
        this(consumer, StringCollectorFactory.INSTANCE.create());
    }

    public ValueContentHandler(
        final GenericConsumer<Object, ? extends JsonException> consumer,
        final StringCollector stringCollector)
    {
        this(consumer, stringCollector, BasicContainerFactory.INSTANCE);
    }

    public ValueContentHandler(
        final GenericConsumer<Object, ? extends JsonException> consumer,
        final StringCollector stringCollector,
        final ContainerFactory containerFactory)
    {
        this.consumer = consumer;
        this.stringCollector = stringCollector;
        this.containerFactory = containerFactory;
    }

    public static JsonParser prepareParser(
        final GenericConsumer<Object, ? extends JsonException> consumer)
    {
        return new JsonParser(
            new StackContentHandler(new ValueContentHandler(consumer)));
    }

    public static Object parse(final String str) throws JsonException {
        BasicGenericConsumer<Object, JsonException> consumer =
            new BasicGenericConsumer<>();
        prepareParser(consumer).parse(str);
        return consumer.get();
    }

    public static Object parse(final Processable<char[]> data)
        throws JsonException
    {
        BasicGenericConsumer<Object, JsonException> consumer =
            new BasicGenericConsumer<>();
        JsonParser parser = prepareParser(consumer);
        data.processWith(parser);
        parser.eof();
        return consumer.get();
    }

    @Override
    public void startObject() {
        stackContentHandler.push(
            new ObjectContentHandler(
                consumer,
                containerFactory.createObjectContainer(),
                stringCollector,
                containerFactory));
    }

    @Override
    public void startArray() {
        List<Object> list = containerFactory.createArrayContainer();
        stackContentHandler.push(
            new ArrayContentHandler(
                new ValueConsumer(list),
                stringCollector,
                containerFactory,
                () -> consumer.accept(list)));
    }

    // CSOFF: ParameterNumber
    @Override
    public void value(
        final char[] buf,
        final int off,
        final int len,
        final boolean eol)
        throws JsonException
    {
        if (eol) {
            consumer.accept(stringCollector.appendLast(buf, off, len));
        } else {
            stringCollector.append(buf, off, len);
        }
    }
    // CSON: ParameterNumber

    @Override
    public void value(final long value) throws JsonException {
        consumer.accept(value);
    }

    @Override
    public void value(final double value) throws JsonException {
        consumer.accept(value);
    }

    @Override
    public void value(final boolean value) throws JsonException {
        consumer.accept(value);
    }

    @Override
    public void nullValue() throws JsonException {
        consumer.accept(null);
    }

    private static class ValueConsumer
        implements GenericConsumer<Object, JsonException>
    {
        private final List<Object> list;

        ValueConsumer(final List<Object> list) {
            this.list = list;
        }

        @Override
        public void accept(final Object value) {
            list.add(value);
        }
    }
}

