package ru.yandex.json.parser;

import java.io.CharArrayWriter;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;

import ru.yandex.json.dom.CopyHandler;
import ru.yandex.json.writer.HumanReadableJsonWriter;
import ru.yandex.json.writer.JsonWriter;

public final class JsonParserBenchmark {
    private static final int BUFFER_SIZE = 65534;
    private static final int REPS = 1000;
    private static final int TESTS = 20;
    private static final int ALL_TYPES = 7;

    private JsonParserBenchmark() {
    }

    public static void main(final String... args) throws Exception {
        int types = ALL_TYPES;
        if (args.length == 1) {
            types = Integer.parseInt(args[0]);
        }
        CharArrayWriter plainWriter = new CharArrayWriter();
        CharArrayWriter dollarWriter = new CharArrayWriter();
        CharArrayWriter humanReadableWriter = new CharArrayWriter();
        try (Reader reader = new InputStreamReader(
                JsonParserBenchmark.class.getResourceAsStream(
                    "big.dollar.json"),
                StandardCharsets.UTF_8);
            JsonWriter plainJsonWriter = JsonWriter.create(plainWriter);
            JsonWriter humanReadableJsonWriter =
                new HumanReadableJsonWriter(humanReadableWriter))
        {
            char[] buf = new char[BUFFER_SIZE];
            JsonParser plainParser =
                new JsonParser(new CopyHandler(plainJsonWriter));
            JsonParser humanReadableParser =
                new JsonParser(new CopyHandler(humanReadableJsonWriter));
            for (int r = reader.read(buf); r != -1; r = reader.read(buf)) {
                dollarWriter.write(buf, 0, r);
                plainParser.process(buf, 0, r);
                humanReadableParser.process(buf, 0, r);
            }
            plainParser.eof();
            humanReadableParser.eof();
        }
        char[] plain = plainWriter.toCharArray();
        char[] dollar = dollarWriter.toCharArray();
        char[] human = humanReadableWriter.toCharArray();
        long min = Long.MAX_VALUE;
        for (int j = 0; j < TESTS; ++j) {
            long start = System.currentTimeMillis();
            for (int i = 0; i < REPS; ++i) {
                if ((types & 1) != 0) {
                    new JsonParser(EmptyContentHandler.INSTANCE)
                        .process(plain);
                }
                if ((types & 2) != 0) {
                    new JsonParser(EmptyContentHandler.INSTANCE)
                        .process(dollar);
                }
                if ((types & (2 + 2)) != 0) {
                    new JsonParser(EmptyContentHandler.INSTANCE)
                        .process(human);
                }
            }
            if (j > 0) {
                long total = System.currentTimeMillis() - start;
                if (total < min) {
                    min = total;
                    --j;
                }
            }
        }
        System.out.println("Best time: " + min);
    }
}

