package ru.yandex.json.parser;

import java.io.IOException;
import java.io.Reader;
import java.util.Arrays;

import ru.yandex.function.CharArrayVoidProcessor;

public class JsonParser implements CharArrayVoidProcessor<JsonException> {
    private static final int HEX_SHIFT = 4;
    private static final char[] ZERO_CHARACTER = new char[1];
    private static final byte[] HEX = {
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        0, 1, 2, 3, 4, 5, 6, 7, 8, 9, -1, -1, -1, -1, -1, -1,
        -1, 10, 11, 12, 13, 14, 15, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, 10, 11, 12, 13, 14, 15
    };

    private static final int INITIAL_STACK_SIZE = 16;
    private static final long TEN = 10L;
    private static final int BUFFER_SIZE = 4096;

    // possible token states
    private static final int NO_TOKEN = 0;
    // dollar string states
    private static final int DOLLAR_STRING = 1;
    private static final int DOLLAR_STRING_ZERO = 2;
    // regular string token states
    private static final int STRING = 3;
    private static final int UNICODE = 4;
    private static final int UNICODE1 = 5;
    private static final int UNICODE2 = 6;
    private static final int UNICODE3 = 7;
    private static final int SLASH = 8;
    // number value states
    private static final int MINUS = 9;
    private static final int ZERO = 10;
    private static final int INTEGER = 11;
    private static final int DOT = 12;
    private static final int FRAC = 13;
    private static final int E = 14;
    private static final int E_SIGN = 15;
    private static final int EXP = 16;
    // null value states
    private static final int N = 'n';
    private static final int U_NULL = 17;
    private static final int L_NULL = 18;
    // true value states
    private static final int T = 't';
    private static final int R = 19;
    private static final int U_TRUE = 20;
    // false value states
    private static final int F = 'f';
    private static final int A = 21;
    private static final int L_FALSE = 22;
    private static final int S = 23;

    // token types
    private static final int STRING_TOKEN = 1;
    private static final int PRIMITIVE = 2;
    private static final int OPEN_CURLY = 3;
    private static final int CLOSE_CURLY = 4;
    private static final int OPEN_SQUARE = 5;
    private static final int CLOSE_SQUARE = 6;
    private static final int COMMA = 7;
    private static final int COLON = 8;

    // parser state
    private static final int START = 9;
    private static final int END = 10;
    // object related states
    private static final int OPEN_OBJECT = 1;
    private static final int AFTER_KEY = 2;
    private static final int AFTER_COLON = 3;
    private static final int AFTER_OBJECT_ENTRY = 4;
    private static final int AFTER_OBJECT_COMMA = 5;
    // array related states
    private static final int OPEN_ARRAY = 6;
    private static final int AFTER_ARRAY_ELEMENT = 7;
    private static final int AFTER_ARRAY_COMMA = 8;

    private static final int[] SLASH_MAP = new int['u' + 1];

    static {
        SLASH_MAP['b'] = '\b';
        SLASH_MAP['f'] = '\f';
        SLASH_MAP['n'] = '\n';
        SLASH_MAP['r'] = '\r';
        SLASH_MAP['t'] = '\t';
        SLASH_MAP['u'] = -1;
        SLASH_MAP['"'] = '"';
        SLASH_MAP['/'] = '/';
        SLASH_MAP['\\'] = '\\';
    }

    private final char[] tmpbuf = new char[1];
    private final ContentHandler handler;
    private final boolean autoReset;

    private int state = START;
    private int tokenState = NO_TOKEN;
    private int tokenType = NO_TOKEN;
    private int[] stack = new int[INITIAL_STACK_SIZE];
    private int stackSize = 0;
    private char unicodeChar = 0;

    // number parsing fields
    private boolean minus;
    private long integer;
    private long frac;
    private int fracLength;
    private boolean minusExp;
    private long exp;

    public JsonParser(final ContentHandler handler) {
        this(handler, false);
    }

    public JsonParser(final ContentHandler handler, final boolean autoReset) {
        this.handler = handler;
        this.autoReset = autoReset;
    }

    public void reset() {
        state = START;
        tokenState = NO_TOKEN;
        tokenType = NO_TOKEN;
        stackSize = 0;
        unicodeChar = 0;
        resetNumber();
    }

    private void popState() throws JsonException {
        state = stack[--stackSize];
        if (state == AFTER_OBJECT_ENTRY) {
            handler.endObjectEntry();
        } else if (autoReset && state == END) {
            reset();
        }
    }

    private void pushState() throws JsonException {
        if (stackSize == stack.length) {
            stack = Arrays.copyOf(stack, stackSize << 1);
        }
        stack[stackSize++] = state + 1;
    }

    //CSOFF: FallThrough
    @SuppressWarnings("fallthrough")
    public void eof() throws JsonException {
        if (state == START && tokenType == NO_TOKEN) {
            switch (tokenState) {
                case ZERO:
                case INTEGER:
                case FRAC:
                case EXP:
                    flushNumber();
                    break;
                case DOLLAR_STRING_ZERO:
                    flushStringEol(ZERO_CHARACTER, 0, 0);
                    break;
                case NO_TOKEN:
                    if (autoReset) {
                        //on autoreset we tolerate empty input
                        break;
                    }
                default:
                    throw new MalformedJsonException(tokenState);
            }
            state = END;
        } else if (state != END) {
            throw new MalformedJsonException(state);
        }
    }
    //CSON: FallThrough

    public void parse(final String str) throws JsonException {
        parse(str.toCharArray());
    }

    public void parse(final char[] str) throws JsonException {
        parse(str, 0, str.length);
    }

    public void parse(final char[] str, final int off, final int len)
        throws JsonException
    {
        process(str, 0, len);
        eof();
        state = START;
    }

    public void parse(final Reader reader) throws IOException, JsonException {
        parse(reader, BUFFER_SIZE);
    }

    public void parse(final Reader reader, final int bufferSize)
        throws IOException, JsonException
    {
        char[] buf = new char[bufferSize];
        while (true) {
            int read = reader.read(buf);
            if (read == -1) {
                eof();
                break;
            } else {
                process(buf, 0, read);
            }
        }
        state = START;
    }

    private void resetNumber() {
        minus = false;
        integer = 0;
        frac = 0L;
        fracLength = 0;
        minusExp = false;
        exp = 0L;
    }

    // CSOFF: ReturnCount
    private int nextToken(final char[] buf, final int off, final int threshold)
        throws JsonException
    {
        switch (tokenState) {
            case NO_TOKEN:
                return readToken(buf, off, threshold);
            case DOLLAR_STRING:
                return dollarString(buf, off, threshold);
            case DOLLAR_STRING_ZERO:
                return dollarStringZero(buf, off);
            case STRING:
                return string(buf, off, threshold);
            case SLASH:
                // 'off < threshold' is enforced by process(...)
                char c = buf[off];
                if (c >= SLASH_MAP.length) {
                    throw malformed(off, c);
                }
                int m = SLASH_MAP[c];
                if (m > 0) {
                    flushChar((char) m);
                } else if (m == 0) {
                    throw malformed(off, c);
                } else {
                    tokenState = UNICODE;
                    unicodeChar = 0;
                }
                return off + 1;
            case UNICODE:
            case UNICODE1:
            case UNICODE2:
            case UNICODE3:
                return stringUnicode(buf, off, threshold);
            case MINUS:
            case ZERO:
            case INTEGER:
            case DOT:
            case FRAC:
            case E:
            case E_SIGN:
            case EXP:
                return number(buf, off, threshold);
            default:
                return primitive(buf, off, threshold);
        }
    }
    // CSON: ReturnCount

    private MalformedJsonException malformed(final int pos, final char c) {
        return new MalformedJsonException(state, pos, c);
    }

    private MalformedJsonException malformed(final int pos) {
        return new MalformedJsonException(state, pos, tokenType);
    }

    // CSOFF: FinalParameter
    // CSOFF: ReturnCount
    @SuppressWarnings("IntLongMath")
    private int readToken(final char[] buf, int off, final int threshold)
        throws JsonException
    {
        while (off < threshold) {
            char c = buf[off++];
            switch (c) {
                case ' ':
                case '\n':
                case '\r':
                case '\t':
                    break;
                case '{':
                    tokenType = OPEN_CURLY;
                    return off;
                case '}':
                    tokenType = CLOSE_CURLY;
                    return off;
                case '[':
                    tokenType = OPEN_SQUARE;
                    return off;
                case ']':
                    tokenType = CLOSE_SQUARE;
                    return off;
                case ':':
                    tokenType = COLON;
                    return off;
                case ',':
                    tokenType = COMMA;
                    return off;
                case '"':
                    tokenState = STRING;
                    return string(buf, off, threshold);
                case '$':
                    tokenState = DOLLAR_STRING;
                    return dollarString(buf, off, threshold);
                case 'n':
                case 't':
                case 'f':
                    tokenState = c;
                    return primitive(buf, off, threshold);
                case '-':
                    tokenState = MINUS;
                    resetNumber();
                    integer = 0L;
                    minus = true;
                    return number(buf, off, threshold);
                case '0':
                    tokenState = ZERO;
                    resetNumber();
                    integer = 0L;
                    return number(buf, off, threshold);
                case '1':
                case '2':
                case '3':
                case '4':
                case '5':
                case '6':
                case '7':
                case '8':
                case '9':
                    tokenState = INTEGER;
                    resetNumber();
                    integer = c - '0';
                    return number(buf, off, threshold);
                default:
                    throw malformed(off, c);
            }
        }
        return off;
    }

    // CSOFF: ParameterNumber
    private void flushString(
        final char[] buf,
        final int off,
        final int threshold,
        final boolean eol)
        throws JsonException
    {
        if (state == OPEN_OBJECT || state == AFTER_OBJECT_COMMA) {
            handler.key(buf, off, threshold - off, eol);
        } else {
            handler.value(buf, off, threshold - off, eol);
        }
    }
    // CSON: ParameterNumber

    private void flushStringEol(
        final char[] buf,
        final int off,
        final int threshold)
        throws JsonException
    {
        flushString(buf, off, threshold, true);
        tokenType = STRING_TOKEN;
        tokenState = NO_TOKEN;
    }

    private void flushChar(final char c) throws JsonException {
        tmpbuf[0] = c;
        flushString(tmpbuf, 0, 1, false);
        tokenState = STRING;
    }

    private int dollarStringZero(final char[] buf, final int off)
        throws JsonException
    {
        // 'off < threshold' is enforced by process(...)
        char c = buf[off];
        if (c == 0) {
            flushString(ZERO_CHARACTER, 0, 1, false);
            tokenState = DOLLAR_STRING;
            return off + 1;
        } else {
            flushStringEol(ZERO_CHARACTER, 0, 0);
            return off;
        }
    }

    private int dollarString(final char[] buf, int off, final int threshold)
        throws JsonException
    {
        int start = off;
        while (off < threshold) {
            if (buf[off] == 0) {
                int nextOff = off + 1;
                if (nextOff < threshold) {
                    if (buf[nextOff] == 0) {
                        flushString(buf, start, nextOff, false);
                        off = nextOff + 1;
                        start = off;
                    } else {
                        flushStringEol(buf, start, off);
                        return nextOff;
                    }
                } else {
                    flushString(buf, start, off, false);
                    tokenState = DOLLAR_STRING_ZERO;
                    return nextOff;
                }
            } else {
                ++off;
            }
        }
        flushString(buf, start, off, false);
        return off;
    }

    @SuppressWarnings("NarrowingCompoundAssignment")
    private int stringUnicode(final char[] buf, int off, final int threshold)
        throws JsonException
    {
        while (off < threshold) {
            char c = buf[off];
            if (c >= HEX.length) {
                throw malformed(off, c);
            }
            byte value = HEX[c];
            if (value == -1) {
                throw malformed(off, c);
            }
            unicodeChar <<= HEX_SHIFT;
            unicodeChar |= value;
            ++off;
            if (tokenState++ == UNICODE3) {
                flushChar(unicodeChar);
                break;
            }
        }
        return off;
    }

    private int string(final char[] buf, int off, final int threshold)
        throws JsonException
    {
        int start = off;
        while (off < threshold) {
            char c = buf[off++];
            if (c == '\\') {
                tokenState = SLASH;
                flushString(buf, start, off - 1, false);
                return off;
            } else if (c == '"') {
                flushStringEol(buf, start, off - 1);
                return off;
            }
        }
        flushString(buf, start, off, false);
        return off;
    }

    private void flushNumber() throws JsonException {
        if (frac == 0L && exp == 0L) {
            if (minus) {
                integer *= -1L;
                if (integer > 0L) {
                    throw new MalformedJsonException(
                        "Integer underflow occured");
                } else {
                    handler.value(integer);
                }
            } else {
                if (integer < 0L) {
                    String strValue = Long.toUnsignedString(integer);
                    char[] chars = strValue.toCharArray();
                    handler.value(chars, 0, chars.length, true);
                } else {
                    handler.value(integer);
                }
            }
        } else {
            StringBuilder sb = new StringBuilder();
            if (minus) {
                sb.append('-');
            }
            if (integer >= 0L) {
                sb.append(integer);
            } else {
                sb.append(Long.toUnsignedString(integer));
            }
            sb.append('.');
            String frac;
            if (this.frac >= 0L) {
                frac = Long.toString(this.frac);
            } else {
                frac = Long.toUnsignedString(this.frac);
            }
            int zeroes = fracLength - frac.length();
            while (zeroes-- > 0) {
                sb.append('0');
            }
            sb.append(frac);
            sb.append('e');
            if (minusExp) {
                sb.append('-');
            }
            sb.append(exp);
            handler.value(Double.parseDouble(new String(sb)));
        }
        tokenType = PRIMITIVE;
        tokenState = NO_TOKEN;
    }

    // CSOFF: MethodLength
    @SuppressWarnings("IntLongMath")
    private int number(final char[] buf, int off, final int threshold)
        throws JsonException
    {
        while (off < threshold) {
            char c = buf[off++];
            switch (tokenState) {
                case MINUS:
                    switch (c) {
                        case '0':
                            tokenState = ZERO;
                            break;
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                            tokenState = INTEGER;
                            integer = c - '0';
                            break;
                        default:
                            throw malformed(off, c);
                    }
                    break;
                case ZERO:
                    switch (c) {
                        case '.':
                            tokenState = DOT;
                            break;
                        case 'E':
                        case 'e':
                            tokenState = E;
                            break;
                        default:
                            flushNumber();
                            return off - 1;
                    }
                    break;
                case INTEGER:
                    switch (c) {
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                            long mul = integer * TEN;
                            long newInteger = mul + c - '0';
                            if (Long.compareUnsigned(mul, newInteger) <= 0
                                && Long.divideUnsigned(mul, TEN) == integer)
                            {
                                integer = newInteger;
                            } else {
                                throw new MalformedJsonException(
                                    off,
                                    "Integer overflow");
                            }
                            break;
                        case '.':
                            tokenState = DOT;
                            break;
                        case 'E':
                        case 'e':
                            tokenState = E;
                            break;
                        default:
                            flushNumber();
                            return off - 1;
                    }
                    break;
                case DOT:
                    switch (c) {
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                            tokenState = FRAC;
                            fracLength = 1;
                            frac = c - '0';
                            break;
                        default:
                            throw malformed(off, c);
                    }
                    break;
                case FRAC:
                    switch (c) {
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                            long mul = frac * TEN;
                            long newFrac = mul + c - '0';
                            if (Long.compareUnsigned(mul, newFrac) <= 0
                                && Long.divideUnsigned(mul, TEN) == frac)
                            {
                                // No integer overflow occured
                                ++fracLength;
                                frac = newFrac;
                            }
                            break;
                        case 'E':
                        case 'e':
                            tokenState = E;
                            break;
                        default:
                            flushNumber();
                            return off - 1;
                    }
                    break;
                case E:
                    switch (c) {
                        case '+':
                            tokenState = E_SIGN;
                            break;
                        case '-':
                            tokenState = E_SIGN;
                            minusExp = true;
                            break;
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                            tokenState = EXP;
                            exp = c - '0';
                            break;
                        default:
                            throw malformed(off, c);
                    }
                    break;
                case E_SIGN:
                    switch (c) {
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                            exp = c - '0';
                            tokenState = EXP;
                            break;
                        default:
                            throw malformed(off, c);
                    }
                    break;
                default: // EXP
                    switch (c) {
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                            long mul = exp * TEN;
                            long newExp = mul + c - '0';
                            if (newExp >= 0L && mul / TEN == exp) {
                                // No integer overflow occured
                                exp = newExp;
                            } else {
                                exp = Long.MAX_VALUE;
                            }
                            break;
                        default:
                            flushNumber();
                            return off - 1;
                    }
                    break;
            }
        }
        return off;
    }
    // CSON: MethodLength

    private int primitive(final char[] buf, int off, final int threshold)
        throws JsonException
    {
        while (off < threshold) {
            char c = buf[off++];
            switch (c) {
                case 'u':
                    if (tokenState == N) {
                        tokenState = U_NULL;
                    } else if (tokenState == R) {
                        tokenState = U_TRUE;
                    } else {
                        throw malformed(off, c);
                    }
                    break;
                case 'l':
                    switch (tokenState) {
                        case U_NULL:
                            tokenState = L_NULL;
                            break;
                        case L_NULL:
                            tokenState = NO_TOKEN;
                            tokenType = PRIMITIVE;
                            handler.nullValue();
                            return off;
                        case A:
                            tokenState = L_FALSE;
                            break;
                        default:
                            throw malformed(off, c);
                    }
                    break;
                case 'r':
                    if (tokenState == T) {
                        tokenState = R;
                    } else {
                        throw malformed(off, c);
                    }
                    break;
                case 'a':
                    if (tokenState == F) {
                        tokenState = A;
                    } else {
                        throw malformed(off, c);
                    }
                    break;
                case 's':
                    if (tokenState == L_FALSE) {
                        tokenState = S;
                    } else {
                        throw malformed(off, c);
                    }
                    break;
                case 'e':
                    boolean value = tokenState == U_TRUE;
                    if (!value && tokenState != S) {
                        throw malformed(off, c);
                    }
                    tokenState = NO_TOKEN;
                    tokenType = PRIMITIVE;
                    handler.value(value);
                    return off;
                default:
                    throw malformed(off, c);
            }
        }
        return off;
    }
    // CSON: ReturnCount

    @Override
    public void process(final char[] buf, int off, final int len)
        throws JsonException
    {
        int threshold = off + len;
        while (off < threshold) {
            off = nextToken(buf, off, len);
            if (tokenType != NO_TOKEN) {
                switch (state) {
                    case START:
                        switch (tokenType) {
                            case STRING_TOKEN:
                            case PRIMITIVE:
                                if (autoReset) {
                                    reset();
                                } else {
                                    state = END;
                                }

                                break;
                            case OPEN_CURLY:
                                pushState();
                                state = OPEN_OBJECT;
                                handler.startObject();
                                break;
                            case OPEN_SQUARE:
                                pushState();
                                state = OPEN_ARRAY;
                                handler.startArray();
                                break;
                            default:
                                throw malformed(off);
                        }
                        break;
                    case OPEN_OBJECT:
                        if (tokenType == STRING_TOKEN) {
                            state = AFTER_KEY;
                        } else if (tokenType == CLOSE_CURLY) {
                            handler.endObject();
                            popState();
                        } else {
                            throw malformed(off);
                        }
                        break;
                    case AFTER_KEY:
                        if (tokenType == COLON) {
                            state = AFTER_COLON;
                        } else {
                            throw malformed(off);
                        }
                        break;
                    case AFTER_COLON:
                        switch (tokenType) {
                            case STRING_TOKEN:
                            case PRIMITIVE:
                                state = AFTER_OBJECT_ENTRY;
                                handler.endObjectEntry();
                                break;
                            case OPEN_CURLY:
                                pushState();
                                state = OPEN_OBJECT;
                                handler.startObject();
                                break;
                            case OPEN_SQUARE:
                                pushState();
                                state = OPEN_ARRAY;
                                handler.startArray();
                                break;
                            default:
                                throw malformed(off);
                        }
                        break;
                    case AFTER_OBJECT_ENTRY:
                        if (tokenType == COMMA) {
                            state = AFTER_OBJECT_COMMA;
                        } else if (tokenType == CLOSE_CURLY) {
                            handler.endObject();
                            popState();
                        } else {
                            throw malformed(off);
                        }
                        break;
                    case AFTER_OBJECT_COMMA:
                        if (tokenType == STRING_TOKEN) {
                            state = AFTER_KEY;
                        } else {
                            throw malformed(off);
                        }
                        break;
                    case OPEN_ARRAY:
                    case AFTER_ARRAY_COMMA:
                        switch (tokenType) {
                            case CLOSE_SQUARE:
                                if (state == OPEN_ARRAY) {
                                    handler.endArray();
                                    popState();
                                } else {
                                    throw malformed(off);
                                }
                                break;
                            case STRING_TOKEN:
                            case PRIMITIVE:
                                state = AFTER_ARRAY_ELEMENT;
                                break;
                            case OPEN_CURLY:
                                state = OPEN_ARRAY;
                                pushState();
                                state = OPEN_OBJECT;
                                handler.startObject();
                                break;
                            case OPEN_SQUARE:
                                state = OPEN_ARRAY;
                                pushState();
                                handler.startArray();
                                break;
                            default:
                                throw malformed(off);
                        }
                        break;
                    case AFTER_ARRAY_ELEMENT:
                        if (tokenType == COMMA) {
                            state = AFTER_ARRAY_COMMA;
                        } else if (tokenType == CLOSE_SQUARE) {
                            handler.endArray();
                            popState();
                        } else {
                            throw malformed(off);
                        }
                        break;
                    default:
                        throw malformed(off);
                }
                tokenType = NO_TOKEN;
            }
        }
    }
    // CSON: FinalParameter
}

