package ru.yandex.json.parser;

import java.lang.reflect.Field;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Arrays;

public class StringCollectorFactory {
    public static final StringCollectorFactory INSTANCE =
        new StringCollectorFactory();

    private static final int MAX_CHUNK_SIZE = 4096;
    private static final int MAX_OVERHEAD = 1024;

    private final boolean shared;

    protected StringCollectorFactory() {
        shared = AccessController.doPrivileged(new SharingChecker());
    }

    public StringCollector create() {
        return create(-1L);
    }

    public StringCollector create(final long expectedSize) {
        StringCollector collector;
        if (shared) {
            if (expectedSize >= 0 && expectedSize <= MAX_CHUNK_SIZE) {
                collector = new ContiguousStringCollector(
                    new StringBuilder((int) expectedSize));
            } else {
                collector =
                    new ChunkedStringCollector(MAX_CHUNK_SIZE, MAX_OVERHEAD);
            }
        } else {
            collector = new SparseStringCollector();
        }
        return collector;
    }

    @SuppressWarnings("deprecation")
    private static class SharingChecker implements PrivilegedAction<Boolean> {
        @Override
        public Boolean run() {
            // CSOFF: EmptyBlock
            try {
                char[] mark = "Shared mark".toCharArray();
                StringBuilder sb = new StringBuilder(mark.length);
                sb.append(mark);
                String s1 = new String(sb);
                String s2 = new String(sb);
                Field f = String.class.getDeclaredField("value");
                boolean accessible = f.isAccessible();
                f.setAccessible(true);
                Object v1 = f.get(s1);
                Object v2 = f.get(s2);
                f.setAccessible(accessible);
                if (v1 == v2 && v1 instanceof char[]) {
                    char[] value = (char[]) v1;
                    if (value.length >= mark.length
                        && Arrays.equals(
                            mark,
                            Arrays.copyOf(value, mark.length)))
                    {
                        return Boolean.TRUE;
                    }
                }
            } catch (IllegalAccessException | NoSuchFieldException e) {
            }
            // CSON: EmptyBlock
            return Boolean.FALSE;
        }
    }
}

