package ru.yandex.json.writer;

import java.io.IOException;
import java.io.Writer;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;

public class HumanReadableJsonWriter extends JsonWriter {
    private static final int PRETTY_PRINT_BITS = 52;
    private static final double MAX_PRETTY_PRINT = 1L << PRETTY_PRINT_BITS;
    private static final double MIN_PRETTY_PRINT = -(1L << PRETTY_PRINT_BITS);
    private static final int FRACTION_DIGITS = 20;
    private static final DecimalFormat FORMAT;

    static {
        FORMAT = new DecimalFormat("#", new DecimalFormatSymbols(Locale.ROOT));
        FORMAT.setMaximumFractionDigits(FRACTION_DIGITS);
    }

    private final StringBuilder currentIndent = new StringBuilder();
    private final String indent;
    private boolean writingString = false;
    private DecimalFormat format = null;

    public HumanReadableJsonWriter(final Writer writer) {
        this(writer, "    ");
    }

    public HumanReadableJsonWriter(final Writer writer, final String indent) {
        super(writer);
        this.indent = indent;
    }

    private void increaseIndent() {
        currentIndent.append(indent);
    }

    private void decreaseIndent() {
        currentIndent.setLength(currentIndent.length() - indent.length());
    }

    private void writeIndent() throws IOException {
        writer.append(currentIndent);
    }

    private void writeLf() throws IOException {
        writer.write('\n');
    }

    @Override
    public void startObject() throws IOException {
        super.startObject();
        increaseIndent();
    }

    @Override
    public void endObject() throws IOException {
        decreaseIndent();
        super.endObject();
    }

    @Override
    public void startArray() throws IOException {
        super.startArray();
        increaseIndent();
    }

    @Override
    public void endArray() throws IOException {
        decreaseIndent();
        super.endArray();
    }

    @Override
    public void value(final double value) throws IOException {
        if (value <= MAX_PRETTY_PRINT
            && value >= MIN_PRETTY_PRINT
            && (value >= 1d || value <= -1d))
        {
            if (format == null) {
                format = (DecimalFormat) FORMAT.clone();
            }
            beforeValue();
            writeValue(format.format(value), false);
        } else {
            super.value(value);
        }
    }

    @Override
    protected void writeStartObject() throws IOException {
        if (ppeek() != JsonState.VALUE) {
            writeIndent();
        }
        super.writeStartObject();
        writeLf();
    }

    @Override
    protected void writeEndObject() throws IOException {
        writeLf();
        writeIndent();
        super.writeEndObject();
    }

    @Override
    protected void writeStartArray() throws IOException {
        if (ppeek() != JsonState.VALUE) {
            writeIndent();
        }
        super.writeStartArray();
        writeLf();
    }

    @Override
    protected void writeEndArray() throws IOException {
        writeLf();
        writeIndent();
        super.writeEndArray();
    }

    @Override
    protected void writeStartString() throws IOException {
        if (!writingString && ppeek() == JsonState.ARRAY) {
            writeIndent();
        }
        super.writeStartString();
    }

    @Override
    protected void writeComma() throws IOException {
        super.writeComma();
        writeLf();
    }

    @Override
    protected void writeKey(final char[] cbuf, final int len)
        throws IOException
    {
        writeIndent();
        super.writeKey(cbuf, len);
        writer.write(' ');
    }

    @Override
    protected void writeValue(
        final char[] value,
        final int len,
        final boolean string)
        throws IOException
    {
        writingString = string;
        if (peek() == JsonState.ARRAY) {
            writeIndent();
        }
        super.writeValue(value, len, string);
        writingString = false;
    }
}

