package ru.yandex.json.writer;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Arrays;
import java.util.Locale;

import ru.yandex.util.unicode.ByteSequence;

public class HumanReadableUtf8JsonWriter extends Utf8JsonWriter {
    private static final int PRETTY_PRINT_BITS = 52;
    private static final double MAX_PRETTY_PRINT = 1L << PRETTY_PRINT_BITS;
    private static final double MIN_PRETTY_PRINT = -(1L << PRETTY_PRINT_BITS);
    private static final int FRACTION_DIGITS = 20;
    private static final DecimalFormat FORMAT;

    static {
        FORMAT = new DecimalFormat("#", new DecimalFormatSymbols(Locale.ROOT));
        FORMAT.setMaximumFractionDigits(FRACTION_DIGITS);
    }

    private final ByteBuffer currentIndent = new ByteBuffer();
    private final ByteBuffer indent;
    private boolean writingString = false;
    private DecimalFormat format = null;

    public HumanReadableUtf8JsonWriter(final OutputStream out) {
        this(out, "    ");
    }

    public HumanReadableUtf8JsonWriter(
        final OutputStream out,
        final String indent)
    {
        super(out);
        this.indent = new ByteBuffer(indent);
    }

    private void increaseIndent() {
        currentIndent.append(indent);
    }

    private void decreaseIndent() {
        currentIndent.length(currentIndent.length() - indent.length());
    }

    private void writeIndent() throws IOException {
        out.write(currentIndent.buffer(), 0, currentIndent.length());
    }

    private void writeLf() throws IOException {
        out.write('\n');
    }

    @Override
    public void startObject() throws IOException {
        super.startObject();
        increaseIndent();
    }

    @Override
    public void endObject() throws IOException {
        decreaseIndent();
        super.endObject();
    }

    @Override
    public void startArray() throws IOException {
        super.startArray();
        increaseIndent();
    }

    @Override
    public void endArray() throws IOException {
        decreaseIndent();
        super.endArray();
    }

    @Override
    public void value(final double value) throws IOException {
        if (value <= MAX_PRETTY_PRINT
            && value >= MIN_PRETTY_PRINT
            && (value >= 1d || value <= -1d))
        {
            if (format == null) {
                format = (DecimalFormat) FORMAT.clone();
            }
            beforeValue();
            writeValue(format.format(value), false);
        } else {
            super.value(value);
        }
    }

    @Override
    protected void writeStartObject() throws IOException {
        if (ppeek() != JsonState.VALUE) {
            writeIndent();
        }
        super.writeStartObject();
        writeLf();
    }

    @Override
    protected void writeEndObject() throws IOException {
        writeLf();
        writeIndent();
        super.writeEndObject();
    }

    @Override
    protected void writeStartArray() throws IOException {
        if (ppeek() != JsonState.VALUE) {
            writeIndent();
        }
        super.writeStartArray();
        writeLf();
    }

    @Override
    protected void writeEndArray() throws IOException {
        writeLf();
        writeIndent();
        super.writeEndArray();
    }

    @Override
    protected void writeStartString() throws IOException {
        if (!writingString && ppeek() == JsonState.ARRAY) {
            writeIndent();
        }
        super.writeStartString();
    }

    @Override
    protected void writeComma() throws IOException {
        super.writeComma();
        writeLf();
    }

    @Override
    protected void writeKey(final char[] cbuf, final int len)
        throws IOException
    {
        writeIndent();
        super.writeKey(cbuf, len);
        out.write(' ');
    }

    @Override
    protected void writeKey(final ByteSequence key) throws IOException {
        writeIndent();
        super.writeKey(key);
        out.write(' ');
    }

    @Override
    protected void writeValue(
        final byte[] value,
        final int offset,
        final int len,
        final boolean string)
        throws IOException
    {
        writingString = string;
        if (peek() == JsonState.ARRAY) {
            writeIndent();
        }
        super.writeValue(value, offset, len, string);
        writingString = false;
    }

    private static class ByteBuffer {
        private byte[] buffer;
        private int length;

        ByteBuffer() {
            buffer = new byte[0];
        }

        ByteBuffer(final String str) {
            buffer = str.getBytes(StandardCharsets.UTF_8);
            length = buffer.length;
        }

        public int length() {
            return length;
        }

        public void length(final int length) {
            this.length = length;
            if (buffer.length < length) {
                buffer = Arrays.copyOf(buffer, length);
            }
        }

        public byte[] buffer() {
            return buffer;
        }

        public void append(final ByteBuffer other) {
            int newLength = length + other.length;
            if (buffer.length < newLength) {
                buffer = Arrays.copyOf(buffer, newLength);
            }
            System.arraycopy(other.buffer, 0, buffer, length, other.length);
            length = newLength;
        }
    }
}

