package ru.yandex.json.writer;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;

import ru.yandex.io.StringBuilderWriter;

public enum JsonType implements JsonWriterFactory {
    NORMAL {
        @Override
        public JsonWriter create(final Writer writer) {
            return new JsonWriter(writer);
        }

        @Override
        public Utf8JsonWriter create(final OutputStream out) {
            return new Utf8JsonWriter(out);
        }
    },
    DOLLAR {
        @Override
        public JsonWriter create(final Writer writer) {
            return new DollarJsonWriter(writer);
        }

        @Override
        public Utf8JsonWriter create(final OutputStream out) {
            return new DollarUtf8JsonWriter(out);
        }
    },
    HUMAN_READABLE {
        @Override
        public JsonWriter create(final Writer writer) {
            return new HumanReadableJsonWriter(writer);
        }

        @Override
        public Utf8JsonWriter create(final OutputStream out) {
            return new HumanReadableUtf8JsonWriter(out);
        }
    },
    COMPATIBILITY {
        @Override
        public JsonWriter create(final Writer writer) {
            return new CompatibilityJsonWriter(writer);
        }

        @Override
        public Utf8JsonWriter create(final OutputStream out) {
            throw new UnsupportedOperationException();
        }
    };

    public void toStringBuilderWriter(
        final StringBuilderWriter sbw,
        final Object value)
    {
        try (JsonWriter writer = create(sbw)) {
            writer.value(value);
        } catch (IOException e) {
            // Impossible case
            throw new RuntimeException(e);
        }
    }

    public void toStringBuilder(final StringBuilder sb, final Object value) {
        toStringBuilderWriter(new StringBuilderWriter(sb), value);
    }

    public String toString(final Object value) {
        StringBuilder sb = new StringBuilder();
        toStringBuilder(sb, value);
        return new String(sb);
    }
}

