package ru.yandex.json.writer;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.Iterator;
import java.util.Map;

public class JsonWriter extends JsonWriterBase {
    private static final char[] UNICODE_PREFIX = "\\u00".toCharArray();
    private static final char[] NULL = "null".toCharArray();
    private static final char[] TRUE = "true".toCharArray();
    private static final char[] FALSE = "false".toCharArray();

    protected final Writer writer;

    public JsonWriter(final Writer writer) {
        this.writer = writer;
    }

    @Override
    public void nullValue() throws IOException {
        beforeValue();
        writeValue(NULL, NULL.length, false);
    }

    @Override
    public void jsonValue(final String value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            beforeValue();
            writer.write(value);
            if (peek() == JsonState.VALUE) {
                --stateLength;
            }
        }
    }

    @Override
    public void value(final int value, final boolean asString)
        throws IOException
    {
        beforeValue();
        sb.setLength(0);
        sb.append(value);
        writeValue(sb, asString);
    }

    @Override
    public void value(final long value, final boolean asString)
        throws IOException
    {
        beforeValue();
        sb.setLength(0);
        sb.append(value);
        writeValue(sb, asString);
    }

    @Override
    public void value(
        final long value,
        final boolean asString,
        final boolean unsigned)
        throws IOException
    {
        beforeValue();
        if (unsigned && value < 0) {
            writeValue(Long.toUnsignedString(value), asString);
        } else {
            sb.setLength(0);
            sb.append(value);
            writeValue(sb, asString);
        }
    }

    @Override
    public void value(final boolean value) throws IOException {
        beforeValue();
        if (value) {
            writeValue(TRUE, TRUE.length, false);
        } else {
            writeValue(FALSE, FALSE.length, false);
        }
    }

    @Override
    public void value(final Throwable value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            startString();
            value.printStackTrace(new PrintWriter(this));
            endString();
        }
    }

    @Override
    public void value(final Object value) throws IOException {
        if (value == null) {
            nullValue();
        } else if (value instanceof String) {
            value((String) value);
        } else if (value instanceof Number) {
            value((Number) value);
        } else if (value instanceof Boolean) {
            value(((Boolean) value).booleanValue());
        } else if (value instanceof Iterable) {
            value((Iterable<?>) value);
        } else if (value instanceof Iterator) {
            value((Iterator<?>) value);
        } else if (value instanceof Map) {
            value((Map<?, ?>) value);
        } else if (value instanceof Throwable) {
            value((Throwable) value);
        } else if (value instanceof JsonValue) {
            value((JsonValue) value);
        } else {
            value(value.toString());
        }
    }

    @Override
    protected void writeStartObject() throws IOException {
        writer.write('{');
    }

    @Override
    protected void writeEndObject() throws IOException {
        writer.write('}');
    }

    @Override
    protected void writeStartArray() throws IOException {
        writer.write('[');
    }

    @Override
    protected void writeEndArray() throws IOException {
        writer.write(']');
    }

    @Override
    protected void writeStartString() throws IOException {
        writer.write('"');
    }

    @Override
    protected void writeEndString() throws IOException {
        writer.write('"');
    }

    @Override
    protected void writeComma() throws IOException {
        writer.write(',');
    }

    @Override
    protected void writeKey(final char[] cbuf, final int len)
        throws IOException
    {
        writeStartString();
        writeImpl(cbuf, 0, len);
        writeEndString();
        writer.write(':');
    }

    @Override
    protected void writeValue(
        final char[] value,
        final int len,
        final boolean string)
        throws IOException
    {
        if (string) {
            writeStartString();
            writeImpl(value, 0, len);
            writeEndString();
        } else {
            writer.write(value, 0, len);
        }
        if (peek() == JsonState.VALUE) {
            --stateLength;
        }
    }

    @Override
    public void close() throws IOException {
        validateState(null);
        writer.close();
    }

    @Override
    public void flush() throws IOException {
        writer.flush();
    }

    @Override
    protected void writeImpl(int c) throws IOException {
        switch (c) {
            case '"':
            case '\\':
                break;

            case '\b':
                c = 'b';
                break;

            case '\f':
                c = 'f';
                break;

            case '\n':
                c = 'n';
                break;

            case '\r':
                c = 'r';
                break;

            case '\t':
                c = 't';
                break;

            default:
                if (c < ' ') {
                    writer.write(UNICODE_PREFIX);
                    if (c > HEX_MASK) {
                        writer.write('1');
                    } else {
                        writer.write('0');
                    }
                    c = HEX[c & HEX_MASK];
                }
                writer.write(c);
                return;
        }
        writer.write('\\');
        writer.write(c);
    }

    @Override
    protected void writeImpl(final char[] cbuf, int off, final int len)
        throws IOException
    {
        int pos = off;
        for (int i = 0; i < len; ++i, ++pos) {
            int c = cbuf[pos];
            if (c < NEED_ESCAPE.length && NEED_ESCAPE[c]) {
                int length = pos - off;
                switch (length) {
                    case 0:
                        break;
                    case 1:
                        writer.write(cbuf[off]);
                        break;
                    default:
                        writer.write(cbuf, off, length);
                        break;
                }
                writeImpl(c);
                off = pos + 1;
            }
        }
        writer.write(cbuf, off, pos - off);
    }
}

