package ru.yandex.json.writer;

import java.io.IOException;
import java.io.Writer;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;

public abstract class JsonWriterBase extends Writer {
    protected static final char[] HEX = "0123456789ABCDEF".toCharArray();
    protected static final int HEX_MASK = 15;
    protected static final boolean[] NEED_ESCAPE = new boolean[] {
        true, true, true, true, true, true, true, true,
        true, true, true, true, true, true, true, true,
        true, true, true, true, true, true, true, true,
        true, true, true, true, true, true, true, true,
        false, false, true, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, true
    };

    private static final char[] EMPTY_CBUF = new char[0];
    private static final int INITIAL_SIZE = 8;

    protected final StringBuilder sb = new StringBuilder();
    private JsonState[] state = new JsonState[INITIAL_SIZE];
    protected int stateLength = 0;
    private boolean[] childs = new boolean[INITIAL_SIZE];
    private int childsLength = 1;
    protected char[] cbuf = EMPTY_CBUF;

    public void reset() {
        stateLength = 0;
        childs[0] = false;
        childsLength = 1;
    }

    public void ensureCbufCapacity(final int len) {
        if (cbuf.length < len) {
            cbuf = new char[Math.max(len, cbuf.length << 1)];
        }
    }

    private IllegalStateException fail(final String expected) {
        return new IllegalStateException("Current JsonWriter states are "
            + Arrays.toString(Arrays.copyOf(state, stateLength))
            + " while state expected to be "
            + expected);
    }

    protected JsonState peek() {
        if (stateLength == 0) {
            return null;
        } else {
            return state[stateLength - 1];
        }
    }

    protected JsonState ppeek() {
        if (stateLength >= 2) {
            return state[stateLength - 2];
        } else {
            return null;
        }
    }

    private void pop() {
        --stateLength;
        if (stateLength > 0 && state[stateLength - 1] == JsonState.VALUE) {
            --stateLength;
        }
    }

    protected void push(final JsonState newState) {
        if (stateLength == state.length) {
            state = Arrays.copyOf(state, stateLength << 1);
        }
        state[stateLength++] = newState;
    }

    protected void validateState(final JsonState expected) {
        if (peek() != expected) {
            throw fail(String.valueOf(expected));
        }
    }

    private void validateValueArrayOrEmptyState() {
        if (stateLength > 0) {
            JsonState state = this.state[stateLength - 1];
            if (state != JsonState.VALUE && state != JsonState.ARRAY) {
                throw fail("VALUE, ARRAY or null");
            }
        }
    }

    private void newChild() {
        if (childsLength == childs.length) {
            childs = Arrays.copyOf(childs, childsLength << 1);
            ++childsLength;
        } else {
            childs[childsLength++] = false;
        }
    }

    protected void putComma() throws IOException {
        if (peek() != JsonState.VALUE) {
            if (childs[childsLength - 1]) {
                writeComma();
            } else {
                childs[childsLength - 1] = true;
            }
        }
    }

    protected abstract void writeComma() throws IOException;

    public void startObject() throws IOException {
        validateValueArrayOrEmptyState();
        putComma();
        newChild();
        push(JsonState.OBJECT);
        writeStartObject();
    }

    protected abstract void writeStartObject() throws IOException;

    public void endObject() throws IOException {
        --childsLength;
        pop();
        writeEndObject();
    }

    protected abstract void writeEndObject() throws IOException;

    public void startArray() throws IOException {
        validateValueArrayOrEmptyState();
        putComma();
        newChild();
        push(JsonState.ARRAY);
        writeStartArray();
    }

    protected abstract void writeStartArray() throws IOException;

    public void endArray() throws IOException {
        --childsLength;
        pop();
        writeEndArray();
    }

    protected abstract void writeEndArray() throws IOException;

    public void startString() throws IOException {
        validateValueArrayOrEmptyState();
        putComma();
        push(JsonState.STRING);
        writeStartString();
    }

    protected abstract void writeStartString() throws IOException;

    public void endString() throws IOException {
        validateState(JsonState.STRING);
        pop();
        writeEndString();
    }

    protected abstract void writeEndString() throws IOException;

    protected void beforeValue() throws IOException {
        validateValueArrayOrEmptyState();
        putComma();
    }

    public void key(final String key) throws IOException {
        validateState(JsonState.OBJECT);
        putComma();
        push(JsonState.VALUE);
        int len = key.length();
        ensureCbufCapacity(len);
        key.getChars(0, len, cbuf, 0);
        writeKey(cbuf, len);
    }

    public void key(final StringBuilder key) throws IOException {
        validateState(JsonState.OBJECT);
        putComma();
        push(JsonState.VALUE);
        int len = key.length();
        ensureCbufCapacity(len);
        key.getChars(0, len, cbuf, 0);
        writeKey(cbuf, len);
    }

    protected abstract void writeKey(final char[] cbuf, final int len)
        throws IOException;

    public abstract void nullValue() throws IOException;

    public abstract void jsonValue(final String value) throws IOException;

    public void value(final float value) throws IOException {
        value(value, false);
    }

    public void value(final double value) throws IOException {
        value(value, false);
    }

    public void value(final int value) throws IOException {
        value(value, false);
    }

    public void value(final long value) throws IOException {
        value(value, false);
    }

    public abstract void value(final int value, boolean asString)
        throws IOException;

    public abstract void value(final long value, boolean asString)
        throws IOException;

    public abstract void value(
        final long value,
        boolean asString,
        boolean unsigned)
        throws IOException;

    public void value(final char value) throws IOException {
        value(Character.toString(value));
    }

    public void value(final String value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            beforeValue();
            writeValue(value, true);
        }
    }

    public void value(final StringBuilder value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            beforeValue();
            writeValue(value, true);
        }
    }

    public void value(final Number value) throws IOException {
        if (value == null) {
            nullValue();
        } else if (value instanceof Double) {
            value(value.doubleValue());
        } else if (value instanceof Float) {
            value(value.floatValue());
        } else if (value instanceof Long
            || value instanceof Integer
            || value instanceof Short
            || value instanceof Byte)
        {
            value(value.longValue());
        } else {
            beforeValue();
            writeValue(value.toString(), false);
        }
    }

    public void value(final double value, final boolean asString)
        throws IOException
    {
        if (Double.isFinite(value)) {
            beforeValue();
            sb.setLength(0);
            sb.append(value);
            writeValue(sb, asString);
        } else {
            nullValue();
        }
    }

    public void value(final float value, final boolean asString)
        throws IOException
    {
        if (Float.isFinite(value)) {
            beforeValue();
            sb.setLength(0);
            sb.append(value);
            writeValue(sb, asString);
        } else {
            nullValue();
        }
    }

    public abstract void value(final boolean value) throws IOException;

    public void value(final Boolean value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            value(value.booleanValue());
        }
    }

    public void value(final Iterable<?> value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            value(value.iterator());
        }
    }

    public void value(final Iterator<?> value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            startArray();
            while (value.hasNext()) {
                value(value.next());
            }
            endArray();
        }
    }

    public void value(final Map<?, ?> value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            startObject();
            for (Map.Entry<?, ?> entry: value.entrySet()) {
                key(entry.getKey().toString());
                value(entry.getValue());
            }
            endObject();
        }
    }

    public abstract void value(final Throwable value) throws IOException;

    public abstract void value(final Object value) throws IOException;

    public void value(final JsonValue value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            value.writeValue(this);
        }
    }

    @Override
    public abstract void close() throws IOException;

    @Override
    public abstract void flush() throws IOException;

    protected void writeValue(final String value, final boolean string)
        throws IOException
    {
        int len = value.length();
        ensureCbufCapacity(len);
        value.getChars(0, len, cbuf, 0);
        writeValue(cbuf, len, string);
    }

    protected void writeValue(final StringBuilder value, final boolean string)
        throws IOException
    {
        int len = value.length();
        ensureCbufCapacity(len);
        value.getChars(0, len, cbuf, 0);
        writeValue(cbuf, len, string);
    }

    protected abstract void writeValue(
        final char[] cbuf,
        final int len,
        final boolean string)
        throws IOException;

    @Override
    public void write(final int i) throws IOException {
        validateState(JsonState.STRING);
        writeImpl(i);
    }

    @Override
    public void write(final char[] cbuf, final int off, final int len)
        throws IOException
    {
        validateState(JsonState.STRING);
        writeImpl(cbuf, off, len);
    }

    @Override
    public void write(final String str, final int off, final int len)
        throws IOException
    {
        validateState(JsonState.STRING);
        writeImpl(str, off, len);
    }

    protected void writeImpl(final String str) throws IOException {
        writeImpl(str, 0, str.length());
    }

    protected void writeImpl(final String str, final int off, final int len)
        throws IOException
    {
        ensureCbufCapacity(len);
        str.getChars(off, off + len, cbuf, 0);
        writeImpl(cbuf, 0, len);
    }

    protected abstract void writeImpl(final int c) throws IOException;

    protected abstract void writeImpl(
        final char[] cbuf,
        final int off,
        final int len)
        throws IOException;
}
