package ru.yandex.json.writer;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;
import java.util.Map;

import ru.yandex.io.StringBuilderWriter;
import ru.yandex.util.unicode.ByteSequence;
import ru.yandex.util.unicode.UnicodeUtil;

public class Utf8JsonWriter extends JsonWriterBase {
    private static final byte[] UNICODE_PREFIX =
        "\\u00".getBytes(StandardCharsets.US_ASCII);
    private static final byte[] EMPTY_BUF = new byte[0];
    private static final byte[] NULL =
        "null".getBytes(StandardCharsets.US_ASCII);
    private static final byte[] TRUE =
        "true".getBytes(StandardCharsets.US_ASCII);
    private static final byte[] FALSE =
        "false".getBytes(StandardCharsets.US_ASCII);
    private static final int NUMBER_CAPACITY = 32;

    private static final boolean[] NEED_ESCAPE = new boolean[] {
        true, true, true, true, true, true, true, true,
        true, true, true, true, true, true, true, true,
        true, true, true, true, true, true, true, true,
        true, true, true, true, true, true, true, true,
        false, false, true, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, true, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false,
        false, false, false, false, false, false, false, false
    };

    protected final OutputStream out;
    private final StringBuilderWriter sbw = new StringBuilderWriter(sb);
    private byte[] buf = EMPTY_BUF;

    public Utf8JsonWriter(final OutputStream out) {
        this.out = out;
    }

    private void ensureBufCapacity(final int len) {
        int requiredLen = len << 2;
        if (buf.length < requiredLen) {
            buf = new byte[Math.max(requiredLen, buf.length << 1)];
        }
    }

    public void key(final ByteSequence key) throws IOException {
        validateState(JsonState.OBJECT);
        putComma();
        push(JsonState.VALUE);
        writeKey(key);
    }

    @Override
    public void nullValue() throws IOException {
        beforeValue();
        writeValue(NULL, NULL.length, false);
    }

    @Override
    public void jsonValue(final String value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            beforeValue();
            int len = value.length();
            ensureCbufCapacity(len);
            value.getChars(0, len, cbuf, 0);
            writeChars(cbuf, 0, len);
            if (peek() == JsonState.VALUE) {
                --stateLength;
            }
        }
    }

    @Override
    public void value(final int value, final boolean asString)
        throws IOException
    {
        beforeValue();
        ensureBufCapacity(NUMBER_CAPACITY);
        int len = UnicodeUtil.toUtf8(value, buf, 0);
        writeValue(buf, len, asString);
    }

    @Override
    public void value(final long value, final boolean asString)
        throws IOException
    {
        beforeValue();
        ensureBufCapacity(NUMBER_CAPACITY);
        int len = UnicodeUtil.toUtf8(value, buf, 0);
        writeValue(buf, len, asString);
    }

    @Override
    public void value(
        final long value,
        final boolean asString,
        final boolean unsigned)
        throws IOException
    {
        beforeValue();
        if (unsigned && value < 0) {
            writeValue(Long.toUnsignedString(value), asString);
        } else {
            ensureBufCapacity(NUMBER_CAPACITY);
            int len = UnicodeUtil.toUtf8(value, buf, 0);
            writeValue(buf, len, asString);
        }
    }

    @Override
    public void value(final boolean value) throws IOException {
        beforeValue();
        if (value) {
            writeValue(TRUE, TRUE.length, false);
        } else {
            writeValue(FALSE, FALSE.length, false);
        }
    }

    public void value(final ByteSequence seq) throws IOException {
        if (seq == null) {
            nullValue();
        } else {
            beforeValue();
            writeValue(seq.bytes(), seq.offset(), seq.length(), true);
        }
    }

    public void value(final byte[] bytes) throws IOException {
        value(bytes, 0, bytes.length);
    }

    public void value(
        final byte[] bytes,
        final int off,
        final int len)
        throws IOException
    {
        if (bytes == null) {
            nullValue();
        } else {
            beforeValue();
            writeValue(bytes, off, len, true);
        }
    }

    @Override
    public void value(final Throwable value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            sb.setLength(0);
            value.printStackTrace(sbw);
            writeValue(sb, true);
        }
    }

    @SuppressWarnings("overloads")
    public void value(final Utf8JsonValue value) throws IOException {
        if (value == null) {
            nullValue();
        } else {
            value.writeValue(this);
        }
    }

    @Override
    public void value(final Object value) throws IOException {
        if (value == null) {
            nullValue();
        } else if (value instanceof String) {
            value((String) value);
        } else if (value instanceof Number) {
            value((Number) value);
        } else if (value instanceof Boolean) {
            value(((Boolean) value).booleanValue());
        } else if (value instanceof Iterable) {
            value((Iterable<?>) value);
        } else if (value instanceof Iterator) {
            value((Iterator<?>) value);
        } else if (value instanceof Map) {
            value((Map<?, ?>) value);
        } else if (value instanceof Throwable) {
            value((Throwable) value);
        } else if (value instanceof ByteSequence) {
            value((ByteSequence) value);
        } else if (value instanceof Utf8JsonValue) {
            value((Utf8JsonValue) value);
        } else if (value instanceof JsonValue) {
            value((JsonValue) value);
        } else {
            value(value.toString());
        }
    }

    @Override
    protected void writeStartObject() throws IOException {
        out.write('{');
    }

    @Override
    protected void writeEndObject() throws IOException {
        out.write('}');
    }

    @Override
    protected void writeStartArray() throws IOException {
        out.write('[');
    }

    @Override
    protected void writeEndArray() throws IOException {
        out.write(']');
    }

    @Override
    protected void writeStartString() throws IOException {
        out.write('"');
    }

    @Override
    protected void writeEndString() throws IOException {
        out.write('"');
    }

    @Override
    protected void writeComma() throws IOException {
        out.write(',');
    }

    @Override
    protected void writeKey(final char[] cbuf, final int len)
        throws IOException
    {
        writeStartString();
        writeImpl(cbuf, 0, len);
        writeEndString();
        out.write(':');
    }

    protected void writeKey(final ByteSequence key) throws IOException {
        writeStartString();
        writeImpl(key.bytes(), key.offset(), key.length());
        writeEndString();
        out.write(':');
    }

    protected void writeValue(
        final byte[] value,
        final int len,
        final boolean string)
        throws IOException
    {
        writeValue(value, 0, len, string);
    }

    protected void writeValue(
        final byte[] value,
        final int offset,
        final int len,
        final boolean string)
        throws IOException
    {
        if (string) {
            writeStartString();
            writeImpl(value, offset, len);
            writeEndString();
        } else {
            out.write(value, offset, len);
        }
        if (peek() == JsonState.VALUE) {
            --stateLength;
        }
    }

    @Override
    protected void writeValue(
        final char[] value,
        final int len,
        final boolean string)
        throws IOException
    {
        if (string) {
            writeStartString();
            writeImpl(value, 0, len);
            writeEndString();
        } else {
            ensureBufCapacity(len);
            int outLen = UnicodeUtil.toUtf8(cbuf, 0, len, buf, 0);
            out.write(buf, 0, outLen);
        }
        if (peek() == JsonState.VALUE) {
            --stateLength;
        }
    }

    @Override
    public void close() throws IOException {
        validateState(null);
        out.close();
    }

    @Override
    public void flush() throws IOException {
        out.flush();
    }

    @Override
    protected void writeImpl(int c) throws IOException {
        switch (c) {
            case '"':
            case '\\':
                break;

            case '\b':
                c = 'b';
                break;

            case '\f':
                c = 'f';
                break;

            case '\n':
                c = 'n';
                break;

            case '\r':
                c = 'r';
                break;

            case '\t':
                c = 't';
                break;

            default:
                if (c < ' ') {
                    out.write(UNICODE_PREFIX);
                    if (c > HEX_MASK) {
                        out.write('1');
                    } else {
                        out.write('0');
                    }
                    c = HEX[c & HEX_MASK];
                }
                out.write(c);
                return;
        }

        out.write('\\');
        out.write(c);
    }

    @Override
    protected void writeImpl(final char[] cbuf, int off, final int len)
        throws IOException
    {
        int pos = off;
        for (int i = 0; i < len; ++i, ++pos) {
            int c = cbuf[pos];
            if (c < NEED_ESCAPE.length && NEED_ESCAPE[c]) {
                int length = pos - off;
                switch (length) {
                    case 0:
                        break;
                    default:
                        writeChars(cbuf, off, length);
                        break;
                }
                writeImpl(c);
                off = pos + 1;
            }
        }
        writeChars(cbuf, off, pos - off);
    }

    protected void writeImpl(final byte[] buf, int off, final int len)
        throws IOException
    {
        int pos = off;
        for (int i = 0; i < len; ++i, ++pos) {
            int c = buf[pos] & 0xff;
            if (NEED_ESCAPE[c]) {
                int length = pos - off;
                switch (length) {
                    case 0:
                        break;
                    case 1:
                        out.write(buf[off]);
                        break;
                    default:
                        out.write(buf, off, length);
                        break;
                }
                writeImpl(c);
                off = pos + 1;
            }
        }
        out.write(buf, off, pos - off);
    }

    protected void writeChars(
        final char[] cbuf,
        final int off,
        final int len)
        throws IOException
    {
        ensureBufCapacity(len);
        int outLen = UnicodeUtil.toUtf8(cbuf, off, len, buf, 0);
        out.write(buf, 0, outLen);
    }
}

