package ru.yandex.json.xpath;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.string.BooleanParser;
import ru.yandex.parser.string.EnumParser;

public final class ValueUtils {
    private ValueUtils() {
    }

    public static int asInt(final Object value)
        throws JsonUnexpectedTokenException
    {
        if (value instanceof Number) {
            return ((Number) value).intValue();
        } else if (value instanceof String) {
            try {
                return Integer.parseInt((String) value);
            } catch (Throwable t) {
                throw new JsonUnexpectedTokenException(value, t);
            }
        }
        throw new JsonUnexpectedTokenException(value);
    }

    public static int asInt(final Object value, final int defaultValue)
        throws JsonUnexpectedTokenException
    {
        if (value == null) {
            return defaultValue;
        } else {
            return asInt(value);
        }
    }

    public static long asLong(final Object value)
        throws JsonUnexpectedTokenException
    {
        if (value instanceof Number) {
            return ((Number) value).longValue();
        } else if (value instanceof String) {
            try {
                return Long.parseLong((String) value);
            } catch (Throwable t) {
                throw new JsonUnexpectedTokenException(value, t);
            }
        }
        throw new JsonUnexpectedTokenException(value);
    }

    public static Long asLongOrNull(final Object value)
        throws JsonUnexpectedTokenException
    {
        if (value == null) {
            return null;
        } else {
            return asLong(value);
        }
    }

    public static double asDouble(final Object value)
        throws JsonUnexpectedTokenException
    {
        if (value instanceof Number) {
            return ((Number) value).doubleValue();
        } else if (value instanceof String) {
            try {
                return Double.parseDouble((String) value);
            } catch (Throwable t) {
                throw new JsonUnexpectedTokenException(value, t);
            }
        }
        throw new JsonUnexpectedTokenException(value);
    }

    public static boolean asBoolean(final Object value)
        throws JsonUnexpectedTokenException
    {
        boolean result;
        if (value instanceof Boolean) {
            result = ((Boolean) value).booleanValue();
        } else if (value instanceof Number) {
            long longValue = ((Number) value).longValue();
            if (longValue == 0L) {
                result = false;
            } else if (longValue == 1L) {
                result = true;
            } else {
                throw new JsonUnexpectedTokenException(value);
            }
        } else if (value instanceof String) {
            try {
                result = BooleanParser.INSTANCE.apply((String) value);
            } catch (Throwable t) {
                throw new JsonUnexpectedTokenException(value, t);
            }
        } else {
            throw new JsonUnexpectedTokenException(value);
        }
        return result;
    }

    public static boolean asBooleanOrDefault(
        final Object value,
        final boolean defaultValue) throws JsonUnexpectedTokenException
    {
        if (value == null) {
            return defaultValue;
        } else {
            return asBoolean(value);
        }
    }

    public static String asStringOrNull(final Object value)
        throws JsonUnexpectedTokenException
    {
        return asStringOrDefault(value, null);
    }

    public static String asStringOrDefault(
        final Object value,
        final String defaultValue)
        throws JsonUnexpectedTokenException
    {
        if (value == null) {
            return defaultValue;
        } else {
            return asString(value);
        }
    }

    public static String asString(final Object value)
        throws JsonUnexpectedTokenException
    {
        if (value instanceof String
            || value instanceof Boolean
            || value instanceof Number)
        {
            return value.toString();
        }
        throw new JsonUnexpectedTokenException(value);
    }

    public static <E extends Enum<E>> E asEnum(
        final Class<E> clazz,
        final Object value)
        throws JsonUnexpectedTokenException
    {
        if (value instanceof String) {
            try {
                return new EnumParser<>(clazz).apply((String) value);
            } catch (Throwable t) {
                throw new JsonUnexpectedTokenException(value, t);
            }
        } else {
            throw new JsonUnexpectedTokenException(value);
        }
    }

    public static List<?> asListOrNull(final Object value)
        throws JsonUnexpectedTokenException
    {
        if (value == null) {
            return null;
        } else {
            return asList(value);
        }
    }

    public static List<?> asList(final Object value)
        throws JsonUnexpectedTokenException
    {
        if (value instanceof List) {
            return (List<?>) value;
        }
        throw new JsonUnexpectedTokenException(value);
    }

    public static Object asSingletonList(final Object value)
        throws JsonUnexpectedTokenException
    {
        List<?> list = asList(value);
        if (list.size() == 1) {
            return list.get(0);
        }
        throw new JsonUnexpectedTokenException(value);
    }

    public static Map<?, ?> asMapOrNull(final Object value)
        throws JsonUnexpectedTokenException
    {
        if (value == null) {
            return null;
        } else {
            return asMap(value);
        }
    }

    public static Map<?, ?> asMap(final Object value)
        throws JsonUnexpectedTokenException
    {
        if (value instanceof Map) {
            return (Map<?, ?>) value;
        }
        throw new JsonUnexpectedTokenException(value);
    }

    public static void handle(
        final PrimitiveHandler handler,
        final Object value)
        throws JsonException
    {
        handle(handler, value, new ArrayList<PathComponent>());
    }

    public static void handle(
        final PrimitiveHandler handler,
        final Object value,
        final List<PathComponent> path)
        throws JsonException
    {
        if (value instanceof Iterable) {
            Iterable<?> iterable = (Iterable<?>) value;
            PathComponent comp = new PathComponent(0);
            path.add(comp);
            for (Object object: iterable) {
                handle(handler, object, path);
                comp.increment();
            }
            path.remove(path.size() - 1);
        } else if (value instanceof Map) {
            Map<?, ?> map = (Map<?, ?>) value;
            PathComponent comp = new PathComponent(null);
            path.add(comp);
            for (Map.Entry<?, ?> entry: map.entrySet()) {
                comp.name(ValueUtils.asString(entry.getKey()));
                handle(handler, entry.getValue(), path);
            }
            path.remove(path.size() - 1);
        } else {
            handler.handle(path, value);
        }
    }
}

