package ru.yandex.json.xpath;

import java.util.AbstractList;
import java.util.Arrays;

import ru.yandex.json.parser.ContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonExceptionHandler;
import ru.yandex.json.parser.StringCollector;

public abstract class XPathContentHandlerBase
    extends AbstractList<PathComponent>
    implements ContentHandler, JsonExceptionHandler
{
    private static final int INITIAL_CAPACITY = 16;

    private final StringCollector keyStringCollector;
    private PathComponent[] path = new PathComponent[INITIAL_CAPACITY];
    private int size = 0;

    protected XPathContentHandlerBase(
        final StringCollector keyStringCollector)
    {
        this.keyStringCollector = keyStringCollector;
    }

    protected void afterValue() {
        if (size > 0) {
            path[size - 1].increment();
        }
    }

    private void ensureCapacity() {
        if (size == path.length) {
            path = Arrays.copyOf(path, size << 1);
        }
    }

    // List implementation
    @Override
    public int size() {
        return size;
    }

    @Override
    public PathComponent get(final int i) {
        if (i >= size) {
            throw new IndexOutOfBoundsException(i + " >= " + size);
        }
        return path[i];
    }

    // JsonExceptionHandler implementation
    @Override
    public void handleException(final JsonException e) throws JsonException {
        e.addSuppressed(new Exception("Exception occured at " + this));
        throw e;
    }

    // ContentHandler implementation
    @Override
    public void startObject() throws JsonException {
    }

    @Override
    public void endObject() throws JsonException {
        afterValue();
    }

    @Override
    public void endObjectEntry() throws JsonException {
        --size;
    }

    @Override
    public void startArray() throws JsonException {
        ensureCapacity();
        path[size++] = new PathComponent(0);
    }

    @Override
    public void endArray() throws JsonException {
        --size;
        afterValue();
    }

    // CSOFF: ParameterNumber
    @Override
    public void key(
        final char[] buf,
        final int off,
        final int len,
        final boolean eol)
        throws JsonException
    {
        if (eol) {
            ensureCapacity();
            path[size++] = new PathComponent(
                keyStringCollector.appendLast(buf, off, len));
        } else {
            keyStringCollector.append(buf, off, len);
        }
    }
    // CSON: ParameterNumber
}

