#define _XOPEN_SOURCE 100500
#define __USE_XOPEN_EXTENDED
#define __USE_GNU

#include "lmdb.h"

#include <util/system/compiler.h>

#include <alloca.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <search.h>
#include <jni.h>

#define ERROR_STRING_FORMAT "%s LMDB error: code=%d: %s"
#define FREQ_PAD 10
#define TXN(handle, expr) { \
    MDB_txn *txn; \
    MDB_cursor *cursor = NULL; \
    int rc = mdb_txn_begin(handle->env, NULL, 0, &txn); \
    if (rc) { \
        setLastError(handle, __func__, rc); \
        goto ABORT;\
    } \
    expr; \
    rc = mdb_txn_commit(txn); \
    if (rc) { \
        setLastError(handle, "COMMIT", rc); \
        goto ABORT;\
    } \
    goto SUCCESS; \
ABORT: \
    if (cursor != NULL) { \
        mdb_cursor_close(cursor); \
    } \
    if (rc != MDB_MAP_FULL) { \
        mdb_txn_abort(txn); \
    } \
    return 1; \
SUCCESS: \
    (void)0; \
}

#define CURSOR(handle, db, expr) { \
    rc = mdb_cursor_open(txn, db, &cursor); \
    if (rc) { \
        setLastError(handle, __func__, rc); \
        goto ABORT;\
    } \
    expr; \
    mdb_cursor_close(cursor); \
}

#define MDB_GET(handle, db, key, value) \
    rc = mdb_get(txn, db, key, value); \
    if (rc && rc != MDB_NOTFOUND) { \
        setLastError(handle, __func__, rc); \
        goto ABORT; \
    }

#define CURSOR_GET(handle, key, value, op) \
    rc = mdb_cursor_get(cursor, key, value, op); \
    if (rc && rc != MDB_NOTFOUND) { \
        setLastError(handle, __func__, rc); \
        goto ABORT; \
    }

#define MDB_PUT(handle, db, key, value, flags) \
    rc = mdb_put(txn, db, key, value, flags); \
    if (rc && rc != MDB_NOTFOUND) { \
        setLastError(handle, __func__, rc); \
        goto ABORT; \
    }

#define MDB_DEL(handle, db, key) \
    rc = mdb_del(txn, db, key, NULL); \
    if (rc && rc != MDB_NOTFOUND) { \
        setLastError(handle, __func__, rc); \
        goto ABORT; \
    }


typedef struct {
    MDB_env *env;
    MDB_dbi dataDb;
    MDB_dbi freqDb;
    MDB_dbi freqIdx;
    char *lastError;
    int lastErrorBufferSize;
} CacheHandle_t;

typedef struct __attribute__((__packed__)) {
    int decompressedSize;
    char data[];
} CacheEntry_t;

typedef struct {
    const char **files;
    int count;
    int size;
} Files_t;

int commonInit(CacheHandle_t* ch) {
    (void) ch;
    return 0;
}


void setLastError(
    CacheHandle_t *handle,
    const char *method,
    int code)
{
    const char *errMsg;
    if (handle->env != NULL) {
        errMsg = mdb_strerror(code);
    } else {
        errMsg = "unknown error";
    }
    int expectedSize =
        snprintf(
            NULL,
            0,
            ERROR_STRING_FORMAT,
            method,
            code,
            errMsg) + 1;
    if (handle->lastErrorBufferSize < expectedSize) {
        handle->lastErrorBufferSize = expectedSize << 1;
        handle->lastError =
            (char *) realloc(handle->lastError, handle->lastErrorBufferSize);
    }
    snprintf(
        handle->lastError,
        handle->lastErrorBufferSize,
        ERROR_STRING_FORMAT,
        method,
        code,
        errMsg);
    fprintf(stderr, "%s\n", handle->lastError);
}

int openDatabase(
    CacheHandle_t *ch,
    const char *name,
    long size)
{
//    fprintf(stderr, "size of int: %d\n", sizeof(int));
    int rc = mdb_env_create(&ch->env);
    if (rc) {
        setLastError(ch, "mdb_env_create", rc);
        return 1;
    }
    rc = mdb_env_set_mapsize(ch->env, size);
    if (rc) {
        setLastError(ch, "mdb_env_set_mapsize", rc);
        return 1;
    }
    rc = mdb_env_set_maxdbs(ch->env, 3);
    if (rc) {
        setLastError(ch, "mdb_env_set_maxdbs", rc);
        return 1;
    }
    rc = mdb_env_open(
        ch->env,
        name,
        MDB_NOSUBDIR | MDB_NOMETASYNC | MDB_NOSYNC | MDB_NOTLS | MDB_NORDAHEAD,
        0666);
    if (rc) {
        setLastError(ch, "mdb_env_open", rc);
        return 1;
    }
    MDB_txn *txn;
    rc = mdb_txn_begin(ch->env, NULL, 0, &txn);
    if (rc) {
        setLastError(ch, "mdb_txn_begin", rc);
        return 1;
    }
    rc = mdb_dbi_open(txn, "data", MDB_CREATE, &ch->dataDb);
    if (rc) {
        setLastError(ch, "mdb_dbi_open: data", rc);
        return 1;
    }
    rc = mdb_dbi_open(txn, "freqDb", MDB_CREATE, &ch->freqDb);
    if (rc) {
        setLastError(ch, "mdb_dbi_open: freqDb", rc);
        return 1;
    }
    rc = mdb_dbi_open(txn, "freqIdx", MDB_CREATE, &ch->freqIdx);
    if (rc) {
        setLastError(ch, "mdb_dbi_open: freqIdx", rc);
        return 1;
    }
    rc = mdb_txn_commit(txn);
    if (rc) {
        setLastError(ch, "mdb_txn_commit", rc);
        return 1;
    }
    return 0;
}

int cachePut(
    CacheHandle_t *ch,
    const char *key,
    const void *data,
    int len,
    int decompressedSize,
    int accessFreq)
{
//    fprintf(stderr, "BEGIN\n");
    TXN(ch,
        MDB_val dbKey;
        long keySize = dbKey.mv_size = strlen(key);
        dbKey.mv_data = (void *) key;

        MDB_val freqValue;
        MDB_GET(ch, ch->freqDb, &dbKey, &freqValue);
        int freq = -1;
        if (rc != MDB_NOTFOUND) {
            freq = ((int *)freqValue.mv_data)[0];
        }

        MDB_val freqKey;
        //INT PADDING + ':' + key.size()
        freqKey.mv_size = FREQ_PAD + keySize + 1;
        //+ \0
        freqKey.mv_data = alloca(freqKey.mv_size + 1);
        if (freq != -1) {
            snprintf(freqKey.mv_data, freqKey.mv_size + 1, "%10d:%s", freq, key);
            MDB_DEL(ch, ch->freqIdx, &freqKey);
        }
        freq = accessFreq;
        snprintf(freqKey.mv_data, freqKey.mv_size + 1, "%10d:%s", 1, key);
        MDB_PUT(ch, ch->freqIdx, &freqKey, &dbKey, 0);

        dbKey.mv_size = keySize;
        dbKey.mv_data = (void *) key;

        MDB_val freqData;
        freqData.mv_size = 4;
        freqData.mv_data = (void *)&freq;
        MDB_PUT(ch, ch->freqDb, &dbKey, &freqData, 0);

        MDB_val dbData;
        dbData.mv_size = 4 + len;
//        dbData.mv_data = (void *) malloc(dbData.mv_size);
//        CacheEntry_t *entry = (CacheEntry_t *) dbData.mv_data;
//        entry->decompressedSize = decompressedSize;
//        fprintf(stderr, "offset of: %d\n", offsetof(CacheEntry_t, data));
//        memcpy(entry->data, data, len);
//        MDB_PUT(ch, ch->dataDb, &dbKey, &dbData, 0);
        MDB_PUT(ch, ch->dataDb, &dbKey, &dbData, MDB_RESERVE);
        CacheEntry_t *entry = (CacheEntry_t *) dbData.mv_data;
        entry->decompressedSize = decompressedSize;
        memcpy(entry->data, data, len);
//        free(dbData.mv_data);

    );
//    fprintf(stderr, "END\n");
    return 0;
}

int cacheUpdate(
    CacheHandle_t *ch,
    const char *key)
{
    TXN(ch,
        MDB_val dbKey;
        dbKey.mv_size = strlen(key);
        dbKey.mv_data = (void *) key;

        MDB_val freqValue;

        MDB_GET(ch, ch->freqDb, &dbKey, &freqValue);
        int freq = -1;
        if (rc != MDB_NOTFOUND) {
            freq = ((int *)freqValue.mv_data)[0];
        }

        MDB_val freqKey;
        //INT PADDING + ':' + key.size()
        freqKey.mv_size = FREQ_PAD + dbKey.mv_size + 1;
        //+ \0
        freqKey.mv_data = alloca(freqKey.mv_size + 1);
        if (freq != -1) {
            snprintf(
                freqKey.mv_data,
                freqKey.mv_size + 1,
                "%10d:%s",
                freq,
                key);
            MDB_DEL(ch, ch->freqIdx, &freqKey);
        } else {
            freq = 0;
        }
        freq++;
        snprintf(freqKey.mv_data, freqKey.mv_size + 1, "%10d:%s", freq, key);
        MDB_PUT(ch, ch->freqIdx, &freqKey, &dbKey, 0);
        dbKey.mv_size = strlen(key);
        dbKey.mv_data = (void *) key;

        MDB_val freqData;
        freqData.mv_size = 4;
        freqData.mv_data = (void *)&freq;
        MDB_PUT(ch, ch->freqDb, &dbKey, &freqData, 0);
    );
    return 0;
}

int cacheRemove(
    CacheHandle_t *ch,
    const char *key)
{
    TXN(ch,
        MDB_val dbKey;
        dbKey.mv_size = strlen(key);
        dbKey.mv_data = (void *) key;

        MDB_val freqValue;

        MDB_GET(ch, ch->freqDb, &dbKey, &freqValue);
        int freq = -1;
        if (!rc) {
            freq = ((int *)freqValue.mv_data)[0];
        }

        MDB_val freqKey;
        //INT PADDING + ':' + key.size()
        freqKey.mv_size = FREQ_PAD + dbKey.mv_size + 1;
        //+ \0
        freqKey.mv_data = alloca(freqKey.mv_size + 1);
        if (freq != -1) {
            snprintf(
                freqKey.mv_data,
                freqKey.mv_size + 1,
                "%10d:%s",
                freq,
                key);
            MDB_DEL(ch, ch->freqIdx, &freqKey);
        }
        MDB_DEL(ch, ch->freqDb, &dbKey);

        MDB_DEL(ch, ch->dataDb, &dbKey);
    );
    return 0;
}

int startsWith(MDB_val *key, const char *needle) {
    if (strncmp(key->mv_data, needle, key->mv_size) == 0) {
        return 1;
    }
    return 0;
}

int cacheRemovePrefix(
    CacheHandle_t *ch,
    const char *key)
{
    TXN(ch,
        CURSOR(ch, ch->freqDb,
            MDB_val currentKey;
            MDB_val freqValue;
            currentKey.mv_size = strlen(key);
            currentKey.mv_data = (void *) key;
            CURSOR_GET(ch, &currentKey, &freqValue, MDB_SET_RANGE);
            MDB_val freqKey;
            int freqKeyBufSize = 0;
            while (rc != MDB_NOTFOUND && startsWith(&currentKey, key)) {
                int freq = ((int *)freqValue.mv_data)[0];
                int freqKeySize = FREQ_PAD + currentKey.mv_size + 1;
                if (freqKey.mv_data == NULL) {
                    freqKey.mv_data = malloc(freqKeySize + 1);
                    freqKeyBufSize = freqKeySize;
                } else if (freqKeyBufSize < freqKeySize) {
                    freqKey.mv_data = realloc(freqKey.mv_data, freqKeySize + 1);
                    freqKeyBufSize = freqKeySize;
                }
                freqKey.mv_size = freqKeySize;
                snprintf(
                    freqKey.mv_data,
                    freqKeySize + 1,
                    "%10d:%.*s",
                    freq,
                    (int) currentKey.mv_size,
                    (char *) currentKey.mv_data);
                MDB_DEL(ch, ch->freqIdx, &freqKey);
                MDB_DEL(ch, ch->freqDb, &currentKey);
                MDB_DEL(ch, ch->dataDb, &currentKey);

                CURSOR_GET(ch, &currentKey, &freqValue, MDB_NEXT);
            }
        );
    );
    return 0;
}

int deleteTop(
    CacheHandle_t *ch,
    int count)
{
    TXN(ch,
        CURSOR(ch, ch->freqIdx,
            MDB_val currentKey;
            MDB_val keyValue;
            CURSOR_GET(ch, &currentKey, &keyValue, MDB_FIRST);
            while (rc != MDB_NOTFOUND && count-- > 0) {
                MDB_DEL(ch, ch->freqIdx, &currentKey);
                MDB_DEL(ch, ch->freqDb, &keyValue);
                MDB_DEL(ch, ch->dataDb, &keyValue);

                CURSOR_GET(ch, &currentKey, &keyValue, MDB_NEXT);
            }
        );
    );
    return 0;
}

long mdbStats(MDB_txn *txn, MDB_dbi dbi, const char *db) {
    Y_UNUSED(db);
    MDB_stat stat;
    memset((void *)&stat, 0, sizeof(stat));
    mdb_stat(txn, dbi, &stat);
/*
    fprintf(stderr, "ms_psize: %s %u, ms_depth: %u, ms_branch_pages: %lu, "
        "ms_leaf_pages: %lu, ms_overflow_pages: %lu, ms_entries: %lu\n",
        db,
        stat.ms_psize,
        stat.ms_depth,
        stat.ms_branch_pages,
        stat.ms_leaf_pages,
        stat.ms_overflow_pages,
        stat.ms_entries);
*/
    return (long) stat.ms_psize * (stat.ms_branch_pages + stat.ms_leaf_pages
        + stat.ms_overflow_pages);
}

long cacheSize(CacheHandle_t* ch) {
    long size = 0;
    TXN(ch,
        size = mdbStats(txn, ch->freqDb, "FREQDB");
        size += mdbStats(txn, ch->freqIdx, "FREQ_IDX");
        size += mdbStats(txn, ch->dataDb, "DATA_DB");
    );
//    fprintf(stderr, "Total size: %ld\n", size);
    return size;
}

int itemsCount(CacheHandle_t* ch) {
    int size = 0;
    TXN(ch,
        MDB_stat stat;
        memset((void *)&stat, 0, sizeof(stat));
        mdb_stat(txn, ch->dataDb, &stat);
        size = stat.ms_entries;
    );
//    fprintf(stderr, "Total size: %ld\n", size);
    return size;
}

void addFile(Files_t *files, const char *file) {
    if (files->count == files->size) {
        files->size = files->count * 2;
        fprintf(stderr, "REALLOC: %d -> %d\n", files->count, files->size);
        files->files =
            (const char **) realloc(
                (void *) files->files,
                sizeof(const char *) * files->size);
    }
    files->files[files->count++] = file;
}

int cacheUniqFiles(
    CacheHandle_t* ch,
    Files_t *files)
{
    int maxItemsCount = itemsCount(ch);
    struct hsearch_data htab;
    memset((void *) &htab, 0, sizeof(htab));
    hcreate_r(maxItemsCount, &htab);
    ENTRY e, *ep;
    e.data = 0;

    TXN(ch,
        CURSOR(ch, ch->freqDb,
            MDB_val currentKey;
            MDB_val keyValue;
            CURSOR_GET(ch, &currentKey, &keyValue, MDB_FIRST);
            while (rc != MDB_NOTFOUND) {
                void *sep = memrchr(
                    currentKey.mv_data,
                    ':',
                    currentKey.mv_size);
                if (sep != NULL) {
                    int keySize =
                        ((char *) sep) - ((char *) currentKey.mv_data);
                    const char *key = malloc(keySize + 1);
                    memcpy((void *) key, currentKey.mv_data, keySize);
                    ((char *) key)[keySize] = 0;

                    e.key = (void *) key;
                    rc = hsearch_r(e, FIND, &ep, &htab);
                    if (!rc) {
                        addFile(files, key);
                        hsearch_r(e, ENTER, &ep, &htab);
                    }
                }

                CURSOR_GET(ch, &currentKey, &keyValue, MDB_NEXT);
            }
        );
    );
    hdestroy_r(&htab);
    return 0;
}

int cacheGet(
    CacheHandle_t* ch,
    const char* key,
    void** value,
    int* size,
    int* decompressedSize)
{
    TXN(ch,
        MDB_val dbKey;
        dbKey.mv_size = strlen(key);
        dbKey.mv_data = (void *) key;

        MDB_val data;
        MDB_GET(ch, ch->dataDb, &dbKey, &data);
        if (rc != MDB_NOTFOUND) {
            CacheEntry_t *entry = (CacheEntry_t *) data.mv_data;
            *size = data.mv_size - 4;
            if (*size > 0) {
                *value = malloc(*size);
                memcpy(*value, entry->data, *size);
            }
            *decompressedSize = entry->decompressedSize;
        }
    );
    return 0;
}

/*
int cacheGetOne(
    CacheHandle_t* ch,
    void** value,
    char** key,
    int* size,
    int* decompressedSize,
    int* accessFreq)
{
    int ret = -1;
    int rc = sqlite3_step(ch->getOneStatement);
    if (rc != SQLITE_ROW) {
        if (rc == SQLITE_DONE) {
            *value = NULL;
            *size = -1;
            ret = 0;
            goto exit;
        }
        setLastError(
            ch,
            "cacheGetOne.step",
            rc);
        goto exit;
    }

    *value = NULL;
    const void* valueBlob = sqlite3_column_blob(ch->getOneStatement, 0);
    *size = sqlite3_column_bytes(ch->getOneStatement, 0);
    *decompressedSize = sqlite3_column_int(ch->getOneStatement, 1);
    *accessFreq = sqlite3_column_int(ch->getOneStatement, 2);
    *key = strdup((const char*) sqlite3_column_text(ch->getOneStatement, 3));

    if (valueBlob != NULL && *size != 0) {
        *value = (void *) malloc(*size);
        memcpy(*value, valueBlob, *size);
//        for (int i = 0; i < 100; i++) {
//            fprintf(stderr, " %x", ((char*)*value)[i]);
//        }
//        fprintf(stderr, "\n");
    }

    ret = 0;

exit:
    sqlite3_reset(ch->getOneStatement);

    return ret;
}
*/

CacheHandle_t* initCache(const char* path, jlong size) {
    CacheHandle_t *handle = malloc(sizeof(CacheHandle_t));
    handle->env = NULL;
    handle->lastErrorBufferSize = 1024;
    handle->lastError = (char *) malloc (handle->lastErrorBufferSize);
    handle->lastError[0] = 0;
    if (openDatabase(handle, path, size)) {
        handle->env = NULL;
        goto exit;
    }
exit:
    return handle;
}

void closeHandle(CacheHandle_t* ch) {
    mdb_dbi_close(ch->env, ch->freqDb);
    mdb_dbi_close(ch->env, ch->freqIdx);
    mdb_dbi_close(ch->env, ch->dataDb);
    mdb_env_close(ch->env);
    free(ch->lastError);
    free(ch);
}

void freeBlob(void* blob) {
    free(blob);
}

__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_init
(JNIEnv* env,
jclass class,
jstring dbPath,
jlong size)
{
    (void) class;
    const char *nativeString = (*env)->GetStringUTFChars(env, dbPath, 0);
    CacheHandle_t *handle = initCache(nativeString, size);
    (*env)->ReleaseStringUTFChars(env, dbPath, nativeString);
    if (handle->env == NULL) {
        jclass exClass;
        char *className = "java/io/IOException";
        exClass = (*env)->FindClass(env, className);
        if (exClass == NULL) {
            return 0;
        }
        return (*env)->ThrowNew(env, exClass, handle->lastError);
    } else {
        return (jlong) handle;
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_commonInit
(JNIEnv* env,
jclass class)
{
    (void) class;
    CacheHandle_t *handle = malloc(sizeof(CacheHandle_t));
    handle->lastErrorBufferSize = 1024;
    handle->lastError = (char *) malloc (handle->lastErrorBufferSize);
    handle->lastError[0] = 0;
    int ret = commonInit(handle);
    if (ret) {
        jclass exClass;
        char *className = "java/io/IOException";
        exClass = (*env)->FindClass(env, className);
        (*env)->ThrowNew(env, exClass, handle->lastError);
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_put
(JNIEnv* env,
jclass class,
jlong handle,
jstring key,
jlong value,
jint len,
jint decompressedSize,
jint accessFreq)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    const char *keyChars = (*env)->GetStringUTFChars(env, key, 0);
    int ret =
        cachePut(
            ch,
            keyChars,
            (const void*) value,
            len,
            decompressedSize,
            accessFreq);
    (*env)->ReleaseStringUTFChars(env, key, keyChars);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_update
(JNIEnv* env,
jclass class,
jlong handle,
jstring key)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    const char *keyChars = (*env)->GetStringUTFChars(env, key, 0);
    int ret = cacheUpdate(ch, keyChars);
    (*env)->ReleaseStringUTFChars(env, key, keyChars);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_remove
(JNIEnv* env,
jclass class,
jlong handle,
jstring key)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    const char *keyChars = (*env)->GetStringUTFChars(env, key, 0);
    int ret = cacheRemove(ch, keyChars);
    (*env)->ReleaseStringUTFChars(env, key, keyChars);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_removePrefix
(JNIEnv* env,
jclass class,
jlong handle,
jstring prefix)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    const char *prefixChars = (*env)->GetStringUTFChars(env, prefix, 0);
    int ret = cacheRemovePrefix(ch, prefixChars);
    (*env)->ReleaseStringUTFChars(env, prefix, prefixChars);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jobject JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_get
(JNIEnv* env,
jclass class,
jlong handle,
jstring key)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    const char* keyChars = (*env)->GetStringUTFChars(env, key, 0);
    void* value = 0;
    int size = -1;
    int decompressedSize = 0;
    int ret = cacheGet(ch, keyChars, &value, &size, &decompressedSize);
    (*env)->ReleaseStringUTFChars(env, key, keyChars);
    if (ret) {
        return NULL;
    }

    jobject cls;
    jmethodID constructor;
    jobject object;

    cls =
        (*env)->FindClass(env, "ru/yandex/cache/lmdb/LMDBCacheEntry");
    if (cls == NULL) {
        fprintf(stderr, "class not found\n");
    }
    constructor = (*env)->GetMethodID(env, cls, "<init>", "(IIJ)V");
    if (constructor == NULL) {
        fprintf(stderr, "constructor not found\n");
    }
    object = (*env)->NewObject(
        env,
        cls,
        constructor,
        size,
        decompressedSize,
        (jlong) value);

    return object;
}

/*
__attribute__((__visibility__("default")))
JNIEXPORT jobject JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_getOne
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    char* key = 0;
    void* value = 0;
    int size = -1;
    int decompressedSize = 0;
    int accessFreq = 0;
    int ret = cacheGetOne(
        ch,
        &value,
        &key,
        &size,
        &decompressedSize,
        &accessFreq);
    if (ret) {
        return NULL;
    }

    jobject cls;
    jmethodID constructor;
    jobject object;

    cls =
        (*env)->FindClass(env, "ru/yandex/cache/sqlite/CopyEntry");
    if (cls == NULL) {
        fprintf(stderr, "class not found\n");
    }
    constructor = (*env)->GetMethodID(
        env,
        cls,
        "<init>",
        "(Ljava/lang/String;IIIJ)V");

    if (constructor == NULL) {
        fprintf(stderr, "constructor not found\n");
    }
    object = (*env)->NewObject(
        env,
        cls,
        constructor,
        (*env)->NewStringUTF(env, key),
        size,
        decompressedSize,
        accessFreq,
        (jlong) value);
    free(key);

    return object;
}
*/

__attribute__((__visibility__("default")))
JNIEXPORT jstring JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_lastError
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return (*env)->NewStringUTF(env, ch->lastError);
}

__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_cacheSize
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    (void) env;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return cacheSize(ch);
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_itemsCount
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    (void) env;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return itemsCount(ch);
}

__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL JavaCritical_ru_yandex_cache_lmdb_LMDBCache_cacheSize
(jlong handle)
{
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return cacheSize(ch);
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_deleteTop
(JNIEnv* env,
jclass class,
jlong handle,
jint count)
{
    (void) class;
    (void) env;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return deleteTop(ch, count);
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL JavaCritical_ru_yandex_cache_lmdb_LMDBCache_deleteTop
(jlong handle,
jint count)
{
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return deleteTop(ch, count);
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_closeHandle
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    (void) env;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return closeHandle(ch);
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL JavaCritical_ru_yandex_cache_lmdb_LMDBCache_closeHandle
(jlong handle)
{
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return closeHandle(ch);
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_freeBlob
(JNIEnv* env,
jclass class,
jlong blob)
{
    (void) class;
    (void) env;
    freeBlob((void *) blob);
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL JavaCritical_ru_yandex_cache_lmdb_LMDBCache_freeBlob
(jlong blob)
{
    freeBlob((void *) blob);
}


__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_copyBlob
(JNIEnv* env,
jclass class,
jlong blob,
jint len)
{
    (void) class;
    (void) env;
    if (blob != 0 && len > 0) {
        void* newBlob = (void*) malloc(len);
        memcpy(newBlob, (void*) blob, len);
        return (jlong) newBlob;
    } else {
        return 0;
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL JavaCritical_ru_yandex_cache_lmdb_LMDBCache_copyBlob
(jlong blob, jint len)
{
    if (blob != 0 && len > 0) {
        void* newBlob = (void*) malloc(len);
        memcpy(newBlob, (void*) blob, len);
        return (jlong) newBlob;
    } else {
        return 0;
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT jobjectArray JNICALL Java_ru_yandex_cache_lmdb_LMDBCache_uniqFiles
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    Files_t files;
    files.count = 0;
    files.size = 10;
    files.files = (const char **) malloc(sizeof(const char *) * files.size);
    int rc = cacheUniqFiles(ch, &files);
    jobjectArray ret = NULL;
    if (rc) {
        return ret;
    }
    ret= (jobjectArray) (*env)->NewObjectArray(
        env,
        files.count,
        (*env)->FindClass(env, "java/lang/String"),
        (*env)->NewStringUTF(env, ""));
    int i;
    for (i = 0; i < files.count; i++) {
        (*env)->SetObjectArrayElement(
            env,
            ret,
            i,
            (*env)->NewStringUTF(env, files.files[i]));
        free((void *) files.files[i]);
    }
    free(files.files);
    return ret;
}

/*
int main(int argc, char** argv) {
    sqlite3* db;
    char* zErrMsg = 0;

    if (db == NULL) {
        db = openDatabase(":memory:");
    }

    if (db == NULL) {
        return 1;
    }

    if (createTables(db)) {
        goto ERROR;
    }

    if (putStatement == NULL) {
        if (prepareStatements(db)) {
            goto ERROR;
        }
    }

    int i;
    char *buf = malloc(8192);
    for (i = 0; i < 8192; i++) {
        buf[i] = i & 0xFF;
    }
    char *key = malloc(1024);
    for (i = 0; i < 1000000; i++) {
        long size = cacheSize(db);
        printf("\rsize: %010ld", size);
        snprintf(key, 1023, "/qwe/sdf/zcqwe/%d:%x", i, i);
        cachePut(db, key, buf, 8192 - (i % 1024));
        if (size == -1) {
            goto ERROR;
        }
        if (size > 1 * 1024 * 1024 * 1024) {
            int rc = dropLeast(
                db,
                1 * 1024 * 1024 * 1024
                    - 128 * 1024 * 1024);
            if (rc == -1) {
                goto ERROR;
            }
        }
    }

    int t;
    for (t = 0; t < 100; t++) {
        printf("\nTest #%d\n", t);
        for (i = 0; i < 100000; i += 4) {
            printf("\rsize: %010ld", cacheSize(db));
            snprintf(key, 1023, "/qwe/sdf/zcqwe/%d:%x", i, i);
//            printf("Removing %s\n", key);
            cacheRemove(db, key);
        }

        for (i = 0; i < 100000; i += 4) {
            printf("\rsize: %010ld", cacheSize(db));
            snprintf(key, 1023, "/qwe/sdf/zcqwe/%d:%x", i, i);
            cachePut(db, key, buf, 8192 - (i % 1024));
        }
    }

    sqlite3_close(db);
    return 0;

ERROR:
    sqlite3_close(db);
    return 1;
}
*/
