package ru.yandex.mail.search.logbroker.logger;

import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiFunction;
import java.util.function.Supplier;
import java.util.logging.Level;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.server.AbstractHttpServer;
import ru.yandex.kikimr.persqueue.LogbrokerClientAsyncFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.ProducerStreamClosedException;
import ru.yandex.kikimr.persqueue.producer.async.AsyncProducerConfig;
import ru.yandex.kikimr.persqueue.producer.transport.message.inbound.ProducerInitResponse;
import ru.yandex.kikimr.persqueue.producer.transport.message.inbound.ProducerWriteResponse;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.logbroker.logger.config.ImmutableLogbrokerLoggerConfig;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class BasicLogbrokerLogger
    extends AbstractStatable
    implements LogbrokerLogger, Runnable
{
    private static final int RECONNECT_INTERVAL_CHECK = 10000;

    private final WriteCallback writeCallback;
    private final Supplier<byte[]> sourceIdSupplier =
        new BsConfigIHostSourceIdSupplier();

    private final PrefixedLogger logger;

    private final TimeFrameQueue<Long> recordsTotal;
    private final TimeFrameQueue<Long> recordsSent;
    private final TimeFrameQueue<Long> sendFailed;

    private final ProxyBalancer proxyBalancer;
    private volatile boolean closed = false;
    private final AtomicBoolean needReconnect = new AtomicBoolean(false);
    private final AsyncProducerConfig.Builder producerConfig;
    private final AtomicReference<LogbrokerContext> contextRef;
    private final Thread reconnectingThread = new Thread(this);

    private long lastReconnect = System.currentTimeMillis();

    public BasicLogbrokerLogger(
        final AbstractHttpServer<?, ?> server,
        final ImmutableLogbrokerLoggerConfig config)
        throws IOException
    {
        this(server, Credentials::none, config);
    }

    public BasicLogbrokerLogger(
        final AbstractHttpServer<?, ?> server,
        final Supplier<Credentials> credentialsSupplier,
        final ImmutableLogbrokerLoggerConfig config)
        throws IOException
    {
        this.logger = server.logger().addPrefix("LB_LOGGER_" + config.topic());

        this.proxyBalancer =
            new ProxyBalancer(
                config.logbrokerHost().getHostName(),
                config.logbrokerHost().getPort());

        this.producerConfig =
            AsyncProducerConfig.builder(config.topic(), sourceIdSupplier.get());
        producerConfig.setCredentialsProvider(credentialsSupplier);

        contextRef = new AtomicReference<>(establishConnection());
        writeCallback = new WriteCallback();

        recordsTotal = new TimeFrameQueue<>(server.config().metricsTimeFrame());
        server.registerStater(
            new PassiveStaterAdapter<>(
                recordsTotal,
                new NamedStatsAggregatorFactory<>(
                    "lb-logger-records-total_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        recordsSent = new TimeFrameQueue<>(server.config().metricsTimeFrame());
        server.registerStater(
            new PassiveStaterAdapter<>(
                recordsSent,
                new NamedStatsAggregatorFactory<>(
                     "lb-logger-records-sent_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        sendFailed = new TimeFrameQueue<>(server.config().metricsTimeFrame());
        server.registerStater(
            new PassiveStaterAdapter<>(
                sendFailed,
                new NamedStatsAggregatorFactory<>(
                    "lb-logger-records-send-failed_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
    }

    @Override
    public void run() {
        try {
            while (!closed) {
                if (needReconnect.get()) {
                    LogbrokerContext context = contextRef.get();
                    if (context != null) {
                        context.close();
                    }

                    long ts = System.currentTimeMillis();
                    if (ts - lastReconnect < RECONNECT_INTERVAL_CHECK) {
                        Thread.sleep(
                            RECONNECT_INTERVAL_CHECK - ts + lastReconnect);
                        continue;
                    }

                    logger.info("Reestablishing connection to logbroker");
                    context = establishConnection();
                    if (context != null) {
                        synchronized (this) {
                            contextRef.set(context);
                            needReconnect.set(false);
                        }

                        lastReconnect = System.currentTimeMillis();
                    }
                } else {
                    synchronized (this) {
                        if (!needReconnect.get()) {
                            this.wait(RECONNECT_INTERVAL_CHECK);
                        }
                    }
                }
            }
        } catch (InterruptedException ie) {
            logger.warning(
                "Logbroker reconnecting thread interrupted, exiting");
        }
    }

    private LogbrokerContext establishConnection() {
        AsyncProducer asyncProducer;
        Long maxSeqNo;
        try {
            logger.info("Creating logbroker producer");
            asyncProducer = new LogbrokerClientAsyncFactory(proxyBalancer)
                .asyncProducer(producerConfig.build()).get();
            logger.info("Logbroker producer created");
            ProducerInitResponse initResponse = asyncProducer.init().get();
            logger.info(
                "Session established " + initResponse.getSessionId()
                    + ' ' + initResponse.getTopic() + ' '
                    + initResponse.getMaxSeqNo()
                    + ' ' + initResponse.getPartition());
            maxSeqNo = initResponse.getMaxSeqNo();
        } catch (ExecutionException | InterruptedException e) {
            logger.log(Level.WARNING, "Failed to init logbroker logger", e);
            //throw new IOException("Failed to init logbroker logger", e);
            return null;
        }

        return new LogbrokerContext(asyncProducer, maxSeqNo);
    }

    @Override
    public CompletableFuture<ProducerWriteResponse> write(final byte[] data) {
        recordsTotal.accept(1L);
        LogbrokerContext context = contextRef.get();

        if (context != null && !context.closed()) {
            CompletableFuture<ProducerWriteResponse> future =
                context.asyncProducer.write(
                    data,
                    context.seq().incrementAndGet());

            future.handleAsync(writeCallback);

            if (future.isCompletedExceptionally()) {
                sendFailed.accept(1L);
                try {
                    logger.warning(
                        "Logbroker write error "
                            + Objects.toString(future.get()));
                } catch (InterruptedException | ExecutionException e) {
                    logger.log(Level.WARNING, "Logbroker write error", e);
                }
            }

            return future;
        } else {
            sendFailed.accept(1L);
            CompletableFuture<ProducerWriteResponse> result =
                new CompletableFuture<>();
            result.complete(null);
            return result;
        }
    }

    @Override
    public void start() {
        reconnectingThread.start();
    }

    protected PrefixedLogger systemLogger() {
        return logger;
    }

    @Override
    public void close() throws IOException {
        closed = true;
        reconnectingThread.interrupt();

        LogbrokerContext context = contextRef.get();
        if (context != null) {
            context.close();
        }
    }

    private static final class BsConfigIHostSourceIdSupplier
        implements Supplier<byte[]>
    {
        private final byte[] sourceId;

        private BsConfigIHostSourceIdSupplier() {
            String host = System.getenv("BSCONFIG_IHOST");
            String port = System.getenv("BSCONFIG_IPORT");

            sourceId = Base64.getEncoder().encode(
                (host + '_' + port).getBytes(StandardCharsets.UTF_8));
        }

        @Override
        public byte[] get() {
            return sourceId;
        }
    }

    private final class WriteCallback
        implements BiFunction<ProducerWriteResponse, Throwable, Object>
    {
        @Override
        public Object apply(
            final ProducerWriteResponse response,
            final Throwable throwable)
        {
            if (throwable != null
                && throwable instanceof ProducerStreamClosedException)
            {
                if (needReconnect.compareAndSet(false, true)) {
                    synchronized (this) {
                        if (!needReconnect.get()) {
                            this.notifyAll();
                        }
                    }
                }
            }

            if (response == null || throwable != null) {
                logger.log(Level.WARNING, "Write failed", throwable);
                sendFailed.accept(1L);
            } else {
                recordsSent.accept(1L);
            }

            return response;
        }
    }

    private static final class LogbrokerContext implements Closeable {
        private final AsyncProducer asyncProducer;
        private final AtomicLong seq;
        private volatile boolean closed = false;

        private LogbrokerContext(
            final AsyncProducer asyncProducer,
            final Long seq)
        {
            this.asyncProducer = asyncProducer;
            this.seq = new AtomicLong(seq);
        }

        public AsyncProducer asyncProducer() {
            return asyncProducer;
        }

        public AtomicLong seq() {
            return seq;
        }

        public boolean closed() {
            return closed;
        }

        @Override
        public void close() {
            closed = true;
            asyncProducer.close();
        }
    }
}
