package ru.yandex.logbroker2;

import java.nio.charset.CharacterCodingException;
import java.nio.charset.StandardCharsets;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import ru.yandex.charset.Decoder;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.xpath.PathComponent;
import ru.yandex.json.xpath.PathMatcher;
import ru.yandex.json.xpath.PrimitiveHandler;
import ru.yandex.json.xpath.XPathContentHandler;
import ru.yandex.logbroker2.config.ImmutableFieldConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.xpath.BadXPathException;

public class JsonDataProcessor implements DataProcessor {
    private final FieldInfo[] fields;

    public JsonDataProcessor(final Map<String, ImmutableFieldConfig> fields)
        throws ConfigException
    {
        this.fields = new FieldInfo[fields.size()];
        int i = 0;
        for (Map.Entry<String, ImmutableFieldConfig> entry
            : fields.entrySet())
        {
            String name = entry.getKey();
            ImmutableFieldConfig config = entry.getValue();
            String selector = config.selector();
            try {
                this.fields[i++] =
                    new FieldInfo(
                        name,
                        config.onDuplicate(),
                        PathMatcher.parse(selector));
            } catch (BadXPathException e) {
                throw new ConfigException(
                    "Can't parse selector <" + selector
                    + "> for field " + name,
                    e);
            }
        }
    }

    @Override
    public Map<String, List<String>> processData(final byte[] data)
        throws LBParseException
    {
        FieldHandler[] handlers = new FieldHandler[fields.length];
        for (int i = 0; i < fields.length; ++i) {
            FieldInfo info = fields[i];
            handlers[i] =
                new FieldHandler(
                    info.name,
                    info.matcher,
                    info.createHandler());
        }
        try {
            JsonParser parser =
                new JsonParser(
                    new XPathContentHandler(
                        new DataHandler(handlers)));
            Decoder decoder = new Decoder(StandardCharsets.UTF_8);
            decoder.decode(data);
            decoder.processWith(parser);
            parser.eof();
        } catch (CharacterCodingException | JsonException e) {
            throw new LBParseException("Failed to parse data", e);
        }
        Map<String, List<String>> result = new HashMap<>(handlers.length << 1);
        for (FieldHandler handler: handlers) {
            List<String> values = handler.handler.values();
            if (values != null) {
                result.put(handler.name, values);
            }
        }
        return result;
    }

    private static class FieldInfo {
        private final String name;
        private final FieldDuplicationPolicy onDuplicate;
        private final PathMatcher matcher;

        FieldInfo(
            final String name,
            final FieldDuplicationPolicy onDuplicate,
            final PathMatcher matcher)
        {
            this.name = name;
            this.onDuplicate = onDuplicate;
            this.matcher = matcher;
        }

        public Handler createHandler() {
            switch (onDuplicate) {
                case OVERWRITE:
                    return new OverwritingHandler();
                case IGNORE:
                    return new IgnoringHandler();
                case ERROR:
                    return new ErrorHandler(name);
                case CONCAT:
                    return new ConcatHandler();
                default: // APPEND
                    return new AppendHandler();
            }
        }
    }

    private interface Handler extends PrimitiveHandler {
        List<String> values();
    }

    private static class FieldHandler {
        private final String name;
        private final PathMatcher matcher;
        private final Handler handler;

        private FieldHandler(
            final String name,
            final PathMatcher matcher,
            final Handler handler)
        {
            this.name = name;
            this.matcher = matcher;
            this.handler = handler;
        }
    }

    private static class DataHandler implements PrimitiveHandler {
        private final FieldHandler[] handlers;

        private DataHandler(final FieldHandler[] handlers) {
            this.handlers = handlers;
        }

        @Override
        public void handle(
            final List<PathComponent> path,
            final Object value)
            throws JsonException
        {
            if (value != null) {
                for (FieldHandler handler: handlers) {
                    if (handler.matcher.matches(path)) {
                        handler.handler.handle(path, value);
                    }
                }
            }
        }
    }

    private static class OverwritingHandler
        extends AbstractList<String>
        implements Handler
    {
        protected String value = null;

        @Override
        public void handle(
            final List<PathComponent> path,
            final Object value)
            throws JsonException
        {
            this.value = value.toString();
        }

        @Override
        public int size() {
            if (value == null) {
                return 0;
            } else {
                return 1;
            }
        }

        @Override
        public String get(final int index) {
            if (index > 0 || value == null) {
                throw new NoSuchElementException();
            } else {
                return value;
            }
        }

        @Override
        public List<String> values() {
            if (value == null) {
                return null;
            } else {
                return this;
            }
        }
    }

    private static class IgnoringHandler extends OverwritingHandler {
        @Override
        public void handle(
            final List<PathComponent> path,
            final Object value)
            throws JsonException
        {
            if (this.value == null) {
                super.handle(path, value);
            }
        }
    }

    private static class ErrorHandler extends OverwritingHandler {
        private final String fieldName;

        private ErrorHandler(final String fieldName) {
            this.fieldName = fieldName;
        }

        @Override
        public void handle(
            final List<PathComponent> path,
            final Object value)
            throws JsonException
        {
            if (this.value == null) {
                this.value = value.toString();
            } else {
                throw new JsonException(
                    "Value for field " + fieldName
                    + " already present. Previous value <"
                    + this.value + ">, new value <" + value + '>');
            }
        }
    }

    private static class ConcatHandler extends OverwritingHandler {
        @Override
        public void handle(
            final List<PathComponent> path,
            final Object value)
            throws JsonException
        {
            if (this.value == null) {
                this.value = value.toString();
            } else {
                this.value = this.value + ',' + value.toString();
            }
        }
    }

    private static class AppendHandler implements Handler {
        private final List<String> values = new ArrayList<>();

        @Override
        public void handle(
            final List<PathComponent> path,
            final Object value)
            throws JsonException
        {
            values.add(value.toString());
        }

        @Override
        public List<String> values() {
            if (values.isEmpty()) {
                return null;
            } else {
                return values;
            }
        }
    }
}

